(ns taoensso.nippy.tools
  "Utilities for third-party tools that want to add user-configurable
  Nippy support. Used by Carmine and Faraday."
  (:require [taoensso.nippy :as nippy]))

(defrecord WrappedForFreezing [value opts])
(defn wrapped-for-freezing? [x] (instance? WrappedForFreezing x))
(defn wrap-for-freezing
  "Ensures that given arg (any freezable data type) is wrapped so that
  (tools/freeze <wrapped-arg>) will serialize the arg using given options."
  ([x     ] (wrap-for-freezing x nil))
  ([x opts]
   (if (wrapped-for-freezing? x)
     (if (= (:opts x) opts)
       x
       (WrappedForFreezing. (:value x) opts))
     (WrappedForFreezing. x opts))))

(defn freeze
  "Like `nippy/freeze` but takes options from special argument wrapper when
  present."
  ([x] (freeze x {}))
  ([x {:keys [default-opts]}]
   (if (wrapped-for-freezing? x)
     (nippy/freeze (:value x) (merge default-opts (:opts x)))
     (nippy/freeze x default-opts))))

(comment (freeze (wrap-for-freezing "wrapped")))

(def ^:dynamic *thaw-opts* nil)
(defmacro with-thaw-opts
  "Evaluates body using given options for any automatic deserialization in
  context."
  [opts & body] `(binding [*thaw-opts* ~opts] ~@body))

(defn thaw "Like `nippy/thaw` but takes options from `*thaw-opts*` binding."
  ([ba] (thaw ba {}))
  ([ba {:keys [default-opts]}]
   (nippy/thaw ba (merge default-opts *thaw-opts*))))
