(ns taoensso.nippy.benchmarks
  {:author "Peter Taoussanis"}
  (:require [taoensso.nippy :as nippy :refer (freeze thaw)]
            [taoensso.nippy.utils :as utils]))

;; Remove stuff from stress-data that breaks reader
(def data (dissoc nippy/stress-data :queue :queue-empty :bytes))

(defmacro bench [& body] `(utils/bench 10000 (do ~@body) :warmup-laps 2000))

(defn freeze-reader [x] (binding [*print-dup* false] (pr-str x)))
(defn thaw-reader   [x] (binding [*read-eval* false] (read-string x)))
(def  roundtrip-reader (comp thaw-reader freeze-reader))

(def roundtrip-defaults  (comp thaw freeze))
(def roundtrip-encrypted (comp #(thaw   % {:password [:cached "p"]})
                               #(freeze % {:password [:cached "p"]})))
(def roundtrip-fast      (comp #(thaw   % {})
                               #(freeze % {:compressor nil})))

(defn autobench []
  (println "Benchmarking roundtrips")
  (println "-----------------------")
  (let [results {:defaults  (bench (roundtrip-defaults  data))
                 :encrypted (bench (roundtrip-encrypted data))
                 :fast      (bench (roundtrip-fast      data))}]
    (println results)
    results))

(comment

  (do ; Roundtrip times
    (println "Benching (this can take some time)...")
    (println "-------------------------------------")

    (println
     {:reader
      {:freeze    (bench (freeze-reader data))
       :thaw      (let [frozen (freeze-reader data)] (bench (thaw-reader frozen)))
       :round     (bench (roundtrip-reader data))
       :data-size (count (.getBytes ^String (freeze-reader data) "UTF-8"))}})

    (println
     {:defaults
      {:freeze    (bench (freeze data))
       :thaw      (let [frozen (freeze data)] (bench (thaw frozen)))
       :round     (bench (roundtrip-defaults data))
       :data-size (count (freeze data))}})

    (println
     {:encrypted
      {:freeze    (bench (freeze data {:password [:cached "p"]}))
       :thaw      (let [frozen (freeze data {:password [:cached "p"]})]
                    (bench (thaw frozen {:password [:cached "p"]})))
       :round     (bench (roundtrip-encrypted data))
       :data-size (count (freeze data {:password [:cached "p"]}))}})

    (println
     {:fast
      {:freeze    (bench (freeze data {:compressor nil}))
       :thaw      (let [frozen (freeze data {:compressor nil})]
                    (bench (thaw frozen)))
       :round     (bench (roundtrip-fast data))
       :data-size (count (freeze data {:compressor nil}))}})

    (println "Done! (Time for cake?)"))

  ;;; 13 June 2013: Clojure 1.5.1, Nippy 2.0.0-alpha1
  ;; {:reader    {:freeze 23124, :thaw 26469, :round 47674, :data-size 22923}}
  ;; {:defaults  {:freeze 4007,  :thaw 2520,  :round 6038,  :data-size 12387}}
  ;; {:encrypted {:freeze 5560,  :thaw 3867,  :round 9157,  :data-size 12405}}
  ;; {:fast      {:freeze 3429,  :thaw 2078,  :round 5577,  :data-size 13237}}

  ;;; 11 June 2013: Clojure 1.5.1, Nippy 1.3.0-alpha1
  ;; {:reader    {:freeze 17042, :thaw 31579, :round 48379, :data-size 22954}}
  ;; {:fast      {:freeze 3078,  :thaw 4684,  :round 8117,  :data-size 13274}}
  ;; {:defaults  {:freeze 3810,  :thaw 5295,  :round 9052,  :data-size 12394}}
  ;; {:encrypted {:freeze 5800,  :thaw 6862,  :round 12317, :data-size 12416}}

  ;;; Clojure 1.5.1, Nippy 1.2.1 (+ sorted-set, sorted-map)
  ;; (def data (dissoc data :sorted-set :sorted-map))
  ;; {:reader {:freeze 15037, :thaw 27885, :round 43945},
  ;;  :nippy  {:freeze 3194,  :thaw 4734,  :round 8380}}
  ;; {:reader-size 22975, :defaults-size 12400, :encrypted-size 12400}

  ;;; Clojure 1.4.0, Nippy 1.0.0 (+ tagged-uuid, tagged-date)
  ;; {:reader {:freeze 22595, :thaw 31148, :round 54059}
  ;;  :nippy  {:freeze 3324,  :thaw 3725,  :round 6918}}

  ;;; Clojure 1.3.0, Nippy 0.9.2
  ;; {:reader {:freeze 28505, :thaw 36451, :round 59545},
  ;;  :nippy  {:freeze 3751,  :thaw 4184,  :round 7769}}

  (println (bench (roundtrip data))) ; Snappy implementations
  ;; {:no-snappy [6163 6064 6042 6176] :JNI [6489 6446 6542 6412]
  ;;  :native-array-copy [6569 6419 6414 6590]}
  )