(ns me.shenfeng.http.client
  (:require [clojure.string :as str])
  ;; TODO Remove HttpClientConfig import?
  (:import [me.shenfeng.http.client HttpClientConfig HttpClient
            IResponseHandler TextRespListener RespListener$IFilter RespListener]
           [java.net Proxy Proxy$Type URI]
           me.shenfeng.http.HttpMethod))

(defn- normalize-headers [headers keywordize-headers?]
  (reduce (fn [m [k v]]
            (assoc m (if keywordize-headers? (keyword (str/lower-case k))
                         (str/lower-case k)) v))
          {} headers))

(defonce default-client (atom nil))

(defn init-client "Initializes and returns a new HTTP client."
  ;; TODO HttpClientConfig deprecated in favour of per-request config?
  [] (HttpClient. (HttpClientConfig. 40000 "http-kit/1.3")))

(defn get-default-client "Returns default HTTP client, initializing as neccesary."
  [] (if-let [c @default-client] c (reset! default-client (init-client))))

(defn request*
  "Issues an asychronous HTTP request and returns a promise object to which the
  value of `(callback http-response)` or `(error-callback exception)` will be
  delivered. See also `request`."
  [{:keys [client url method headers data timeout user-agent]
    :or   {method :get timeout 40000 user-agent "http-kit/1.3"
           client (get-default-client)}}
   callback & [error-callback]]
  (let [uri (URI. url)
        method (case method
                 :get  HttpMethod/GET
                 :post HttpMethod/POST
                 :put  HttpMethod/PUT)
        response (promise)]
    ;; TODO Remove proxy, pass timeout, user-agent to HttpClient
    (.exec ^HttpClient client uri method headers data Proxy/NO_PROXY
           (RespListener.
            (reify IResponseHandler
              (onSuccess [this status headers body]
                (deliver response
                         (callback {:body body
                                    :headers (normalize-headers headers true)
                                    :status status})))
              (onThrowable [this t]
                (deliver response
                         (try ((or error-callback callback) (Exception. t))
                              (catch Exception e e)))))))
    response))

(defmacro request
  "Issues an asynchronous HTTP request, binds the HTTP response or exception to
  `resp`, then executes the given handler body in the context of that binding.
  Returns a promise object to which the handler's return value will be delivered:

     ;; Asynchronous
     (request {:url \"http://www.cnn.com/\"}
              {:keys [status body headers] :as resp}
              (if status ; nil on exceptions
                (do (println \"Body: \" body) body)
                (do (println \"Exception: \" resp) resp)))

     ;; Synchronous
     @(request ...) or (deref (request ...) timeout-ms timeout-val)

  See low-level `request*` for options."
  [options resp & handler]
  `(request* ~options (fn [~resp] ~@handler)))

(comment
  @(request {:url "http://www.cnn.com/"}
            {:keys [status body headers] :as resp}
            (if status ; nil on exceptions
              (do (println "Body: " body) body)
              (do (println "Exceptiond: " resp) resp))))