(ns com.syntereen.gsheetj.auth
  (:import
   ;;
   ;; for Google API client
   ;;
   (com.google.api.client.extensions.java6.auth.oauth2 AuthorizationCodeInstalledApp)
   (com.google.api.client.extensions.jetty.auth.oauth2 LocalServerReceiver$Builder)
   (com.google.api.client.googleapis.auth.oauth2 GoogleAuthorizationCodeFlow$Builder)
   (com.google.api.client.googleapis.auth.oauth2 GoogleClientSecrets)
   (com.google.api.client.googleapis.javanet GoogleNetHttpTransport)
   (com.google.api.client.json.jackson2 JacksonFactory)
   (com.google.api.client.util.store FileDataStoreFactory)
   (java.io FileInputStream InputStreamReader File)
   (java.util Collections)
   ;;
   ;; For Google Sheets
   ;;
   (com.google.api.services.sheets.v4 Sheets$Builder)
   (com.google.api.services.sheets.v4 SheetsScopes)
   )
  )

(set! *warn-on-reflection* true)

(defn get-google-sheets-authentication-credential
  "Get authentication credentials for a Google Sheets desktop app.
  If a valid credential for an `app-user` exists, just returns it.
  If not, go through a re-authorization step.
  Note that this also saves the credentials in a data store."
  [http-transport json-factory ^String app-credentials-file-path ^String app-user-tokens-directory-path app-user]
  (let [local-server-port 8888
        scopes (Collections/singletonList SheetsScopes/SPREADSHEETS)
        client-secrets (GoogleClientSecrets/load json-factory (InputStreamReader. (FileInputStream. app-credentials-file-path)))
        data-store-factory (FileDataStoreFactory. (File. app-user-tokens-directory-path))
        gacf (-> (GoogleAuthorizationCodeFlow$Builder. http-transport json-factory client-secrets scopes)
                 (.setDataStoreFactory data-store-factory)
                 (.setAccessType "offline")
                 (.build))
        receiver (-> (LocalServerReceiver$Builder.)
                     (.setPort local-server-port)
                     (.build))
        cred (-> (AuthorizationCodeInstalledApp. gacf receiver)
                 (.authorize app-user))
        ]
    cred))

(defn google-sheets-client
  "Creates a client that can be used to create a google-sheets-service.
  Note that this is credential specific, thus app-user specific."
  [app-user app-credentials-file-path app-user-tokens-directory-path]
  (let [http-transport (GoogleNetHttpTransport/newTrustedTransport)
        json-factory (JacksonFactory/getDefaultInstance)
        credential (get-google-sheets-authentication-credential http-transport
                                                                json-factory
                                                                app-credentials-file-path
                                                                app-user-tokens-directory-path
                                                                app-user)
        ]
    {:credential credential
     :json-factory json-factory
     :http-transport http-transport}))

(defn google-sheets-service
  "Creates a Google Sheets Service that is used in Google Sheets calls."
  [google-sheets-client ^String app-name]
  (let [{:keys [http-transport json-factory credential]} google-sheets-client]
    (-> (Sheets$Builder. http-transport json-factory credential)
        (.setApplicationName app-name)
        (.build))))


(defn google-sheets-service-component
  "For creating a Google Sheets Service in a state management system."
  [{:keys [app-user app-credentials-file-path app-user-tokens-directory-path app-name]}]
  (google-sheets-service (google-sheets-client app-user
                                               app-credentials-file-path
                                               app-user-tokens-directory-path)
                         app-name))
