(ns com.syntereen.gsheetj
  (:import
   ;;
   ;; For Google Sheets
   ;;
   (com.google.api.services.sheets.v4.model ValueRange)
   (com.google.api.services.sheets.v4.model ClearValuesRequest)
   (java.util ArrayList)
   ;;
   ;; for type hints
   ;;
   (com.google.api.services.sheets.v4 Sheets)
   (java.util Collection)
   )
  )

(set! *warn-on-reflection* true)

(defn ->ValueRange
  "Convert a vector (rows) of vectors (cells) into a ValueRange suitable for Google Sheets."
  [vec-of-vecs]
  (let [data (ArrayList. ^Collection (map #(ArrayList. ^Collection (map (fn [x] (if (keyword? x) (str x) x)) %)) vec-of-vecs))
        vr (doto (ValueRange.) (.setValues data))]
    vr))

(defn clear-sheet-range
  "Clears the specified range. Values only.  No formatting is cleared."
  [^Sheets sheets-service spreadsheet-id sheet-and-range]
  (let [cr (ClearValuesRequest.)
        request (-> sheets-service
                    (.spreadsheets)
                    (.values)
                    (.clear spreadsheet-id sheet-and-range cr))
        response (.execute request)]
    response))

(defn get-sheet-range
  "Return the range values as specified.
  `:value-render-option` can be one of: \"FORMATTED_VALUE\" (in the locale of the spreadsheet. Default.),
  \"UNFORMATTED_VALUE\", \"FORMULA\".
  `date-time-render-option` can be one of: \"SERIAL_NUMBER\" (Default.),
  \"FORMATTED_STRING\" (in the locale of the spreadsheet)."
  [^Sheets sheets-service spreadsheet-id sheet-and-range & {:keys [value-render-option date-time-render-option]}]
  (let [request (-> sheets-service
                    (.spreadsheets)
                    (.values)
                    (.get spreadsheet-id sheet-and-range)
                    (cond-> value-render-option (.setValueRenderOption value-render-option)
                            date-time-render-option (.setDateTimeRenderOption date-time-render-option)))
        ^ValueRange response (.execute request)]
    (.getValues response)))

;;; You can choose if you want to overwrite existing data after a table or insert new rows for the new data.
;;; By default, the input overwrites data after the table.
;;; To write the new data into new rows, specify {:insert-data-option "INSERT_ROWS"}.
(defn append-sheet-range
  "Appends the specified range with the values specified.
  `:value-input-option` must be one of: \"RAW\" (as entered),
  \"USER_ENTERED\" (parsed as if entered in the UI).
  `:insert-data-option` must be one of: \"OVERWRITE\", \"INSERT_ROWS\".
  `:include-values-in-response` `false` (Do not return the new values.  Default), `true`.
  `:response-value-render-option` can be one of: \"FORMATTED_VALUE\" (in the locale of the spreadsheet. Default.),
  \"UNFORMATTED_VALUE\", \"FORMULA\".
  `response-date-time-render-option` can be one of: \"SERIAL_NUMBER\" (Default.),
  \"FORMATTED_STRING\" (in the locale of the spreadsheet). Ignored if `:response-value--render-option` is \"FORMATTED_VALUE\"."
  [^Sheets sheets-service spreadsheet-id sheet-and-range data-values
   & {:keys [value-input-option
             insert-data-option
             include-values-in-response
             response-value-render-option
             response-date-time-render-option]}]
  (let [vr (->ValueRange data-values)
        request (-> sheets-service
                    (.spreadsheets)
                    (.values)
                    (.append spreadsheet-id sheet-and-range vr)
                    (cond-> value-input-option (.setValueInputOption value-input-option)
                            insert-data-option (.setInsertDataOption insert-data-option)
                            include-values-in-response (.setIncludeValuesInResponse include-values-in-response)
                            response-value-render-option (.setResponseValueRenderOption response-value-render-option)
                            response-date-time-render-option (.setResponseDateTimeRenderOption response-date-time-render-option)))
        response (.execute request)]
    response))

(defn update-sheet-range
  "Updates the specified range with the values specified.
  `:value-input-option` must be one of: \"RAW\" (as entered),
  \"USER_ENTERED\" (parsed as if entered in the UI).
  `:include-values-in-response` `false` (Do not return the new values.  Default), `true`.
  `:response-value-render-option` can be one of: \"FORMATTED_VALUE\" (in the locale of the spreadsheet. Default.),
  \"UNFORMATTED_VALUE\", \"FORMULA\".
  `response-date-time-render-option` can be one of: \"SERIAL_NUMBER\" (Default.),
  \"FORMATTED_STRING\" (in the locale of the spreadsheet). Ignored if `:response-value--render-option` is \"FORMATTED_VALUE\"."
  [^Sheets sheets-service spreadsheet-id sheet-and-range data-values
   & {:keys [value-input-option
             include-values-in-response
             response-value-render-option
             response-date-time-render-option]}]
  (let [vr (->ValueRange data-values)
        request (-> sheets-service
                    (.spreadsheets)
                    (.values)
                    (.update spreadsheet-id sheet-and-range vr)
                    (cond-> value-input-option (.setValueInputOption value-input-option)
                            include-values-in-response (.setIncludeValuesInResponse include-values-in-response)
                            response-value-render-option (.setResponseValueRenderOption response-value-render-option)
                            response-date-time-render-option (.setResponseDateTimeRenderOption response-date-time-render-option)))
        response (.execute request)]
    response))
