(ns dawdle.web
  (:require
    [aleph.http :as http]
    [cheshire.core :as json]
    [manifold.deferred :as d]
    [clojure.java.io :as io])
  (:import (clojure.lang ExceptionInfo)))

(defn- slack-url [method]
  (format "https://slack.com/api/%s" method))

(defn- parse-json [res]
  (json/parse-stream res true))

(defn- handle-error [res]
  (if (:error res)
    (d/error-deferred (ExceptionInfo. (:error res) res))
    res))

(defn- item-extractor [key res]
  (if key
    (get res key)
    res))

(defn slack-post
  ([method params]
    (slack-post method nil params))
  ([method res-key params]
    (let [url (slack-url method)
          extract-item (partial item-extractor res-key)]
      (println method)
      (d/chain
        (http/post
          url
          {:query-params params})
        :body
        io/reader
        parse-json
        handle-error
        extract-item))))

(defn api-test
  ([] (api-test {}))
  ([params]
    (->> params
      (slack-post "api.test"))))

(defn auth-test
  ([token]
    (->> {:token token}
         (slack-post "auth.test"))))

(defn channels-archive
  ([token channel] (channels-archive token channel {}))
  ([token channel params]
    (->> {:token token :channel channel}
         (conj params)
         (slack-post "channels.archive"))))

(defn channels-create
  ([token name] (channels-create token name {}))
  ([token name params]
    (->> {:token token :name name}
         (conj params)
         (slack-post "channels.create" :channel))))

(defn channels-history
  ([token channel] (channels-history token channel {}))
  ([token channel params]
    (->> {:token token :channel channel}
         (conj params)
         (slack-post "channels.history" :messages))))

(defn channels-info
  ([token channel] (channels-info token channel {}))
  ([token channel params]
    (->> {:token token :channel channel}
         (conj params)
         (slack-post "channels.info" :channel))))

(defn channels-invite
  ([token user channel] (channels-invite token user channel {}))
  ([token user channel params]
    (->> {:token token :user user :channel channel}
         (conj params)
         (slack-post "channels.invite"))))

(defn channels-join
  ([token name] (channels-join token name {}))
  ([token name params]
    (->> {:token token :name name}
         (conj params)
         (slack-post "channels.join"))))

(defn channels-kick
  ([token user channel] (channels-kick token user channel {}))
  ([token user channel params]
    (->> {:token token :user user :channel channel}
         (conj params)
         (slack-post "channels.kick"))))

(defn channels-leave
  ([token channel] (channels-leave token channel {}))
  ([token channel params]
    (->> {:token token :channel channel}
         (conj params)
         (slack-post "channels.leave"))))

(defn channels-list
  ([token] (channels-list token {}))
  ([token params]
    (->> {:token token}
         (conj params)
         (slack-post "channels.list" :channels))))

(defn channels-mark
  ([token channel ts] (channels-mark token channel ts {}))
  ([token channel ts params]
    (->> {:token token :channel channel :ts ts}
         (conj params)
         (slack-post "channels.mark"))))

(defn channels-rename
  ([token channel name] (channels-rename token channel name {}))
  ([token channel name params]
    (->> {:token token :channel channel :name name}
         (conj params)
         (slack-post "channels.rename"))))

(defn channels-set-purpose
  ([token channel purpose] (channels-set-purpose token channel purpose {}))
  ([token channel purpose params]
    (->> {:token token :purpose purpose :channel channel}
         (conj params)
         (slack-post "channels.setPurpose"))))

(defn channels-set-topic
  ([token channel topic] (channels-set-purpose token channel topic {}))
  ([token channel topic params]
    (->> {:token token :topic topic :channel channel}
         (conj params)
         (slack-post "channels.setTopic"))))

(defn channels-unarchive
  ([token channel] (channels-unarchive token channel {}))
  ([token channel params]
    (->> {:token token :channel channel}
         (conj params)
         (slack-post "channels.unarchive"))))

(defn chat-delete
  ([token channel ts params]
    (->> {:token token :channel channel :ts ts}
         (conj params)
         (slack-post "chat.delete"))))

(defn chat-post-message
  ([token channel text] (chat-post-message token channel text {}))
  ([token channel text params]
    (->> {:token token :channel channel :text text}
         (conj params)
         (slack-post "chat.postMessage"))))

(defn chat-update
  ([token channel ts text] (chat-update token channel ts text {}))
  ([token channel ts text params]
   (println {:token token :channel channel :ts ts :text text})
    (->> {:token token :channel channel :ts ts :text text}
         (conj params)
         (slack-post "chat.update"))))

(defn emoji-list
  ([token params]
    (->> {:token token}
         (conj params)
         (slack-post "emoji.list" :emoji))))

(defn files-delete
  ([token file params]
    (->> {:token token :file file}
         (conj params)
         (slack-post "files.delete"))))

(defn files-info
  ([token file params]
    (->> {:token token :file file}
         (conj params)
         (slack-post "files.info"))))

(defn files-list
  ([token params]
    (->> {:token token}
         (conj params)
         (slack-post "files.list"))))

(defn files-upload
  ([token params]
    (->> {:token token}
         (conj params)
         (slack-post "files.list"))))

(defn groups-archive
  ([token group] (groups-archive token group {}))
  ([token group params]
    (->> {:token token :channel group}
         (conj params)
         (slack-post "groups.archive"))))

(defn groups-create
  ([token name] (groups-create token name {}))
  ([token name params]
    (->> {:token token :name name}
         (conj params)
         (slack-post "groups.create" :group))))

(defn groups-history
  ([token group] (groups-history token group {}))
  ([token group params]
    (->> {:token token :channel group}
         (conj params)
         (slack-post "groups.history" :messages))))

(defn groups-info
  ([token group] (groups-info token group {}))
  ([token group params]
    (->> {:token token :channel group}
         (conj params)
         (slack-post "groups.info" :group))))

(defn groups-invite
  ([token user group] (groups-invite token user group {}))
  ([token user group params]
    (->> {:token token :user user :channel group}
         (conj params)
         (slack-post "groups.invite"))))

(defn groups-kick
  ([token user group] (groups-kick token user group {}))
  ([token user group params]
    (->> {:token token :user user :channel group}
         (conj params)
         (slack-post "groups.kick"))))

(defn groups-leave
  ([token group] (groups-leave token group {}))
  ([token group params]
    (->> {:token token :channel group}
         (conj params)
         (slack-post "groups.leave"))))

(defn groups-list
  ([token] (groups-list token {}))
  ([token params]
    (->> {:token token}
         (conj params)
         (slack-post "groups.list" :groups))))

(defn groups-mark
  ([token group ts] (groups-mark token group ts {}))
  ([token group ts params]
    (->> {:token token :channel group :ts ts}
         (conj params)
         (slack-post "groups.mark"))))

(defn groups-rename
  ([token group name] (groups-rename token group name {}))
  ([token group name params]
    (->> {:token token :channel group :name name}
         (conj params)
         (slack-post "groups.rename"))))

(defn groups-set-purpose
  ([token group purpose] (groups-set-purpose token group purpose {}))
  ([token group purpose params]
    (->> {:token token :purpose purpose :channel group}
         (conj params)
         (slack-post "groups.setPurpose"))))

(defn groups-set-topic
  ([token group topic] (groups-set-topic token group topic {}))
  ([token group topic params]
    (->> {:token token :topic topic :channel group}
         (conj params)
         (slack-post "groups.setTopic"))))

(defn groups-unarchive
  ([token group] (groups-unarchive token group {}))
  ([token group params]
    (->> {:token token :channel group}
         (conj params)
         (slack-post "groups.unarchive"))))

(defn im-close
  ([token channel] (im-close token channel {}))
  ([token channel params]
    (->> {:token token :channel channel}
         (conj params)
         (slack-post "im.close"))))

(defn im-history
  ([token channel params]
    (->> {:token token :channel channel}
         (conj params)
         (slack-post "im.history" :messages))))

(defn im-info
  ([token channel] (im-info token channel {}))
  ([token channel params]
    (->> {:token token :channel channel}
         (conj params)
         (slack-post "im.info" :im))))

(defn im-list
  ([token] (im-list token {}))
  ([token params]
    (->> {:token token}
         (conj params)
         (slack-post "im.list" :ims))))

(defn im-mark
  ([token channel ts] (im-mark token channel ts {}))
  ([token channel ts params]
    (->> {:token token :channel channel :ts ts}
         (conj params)
         (slack-post "im.list"))))

(defn im-open
  ([token user] (im-open token user {}))
  ([token user params]
    (->> {:token token :user user}
         (conj params)
         (slack-post "im.open" :channel))))

(defn mpim-close
  ([token channel params]
    (->> {:token token :channel channel}
         (conj params)
         (slack-post "mpim.close"))))

(defn mpim-history
  ([token channel params]
    (->> {:token token :channel channel}
         (conj params)
         (slack-post "mpim.history" :messages))))

(defn mpim-info
  ([token channel params]
    (->> {:token token :channel channel}
         (conj params)
         (slack-post "mpim.info" :mpim))))

(defn mpim-list
  ([token params]
    (->> {:token token}
         (conj params)
         (slack-post "mpim.list" :mpims))))

(defn mpim-mark
  ([token channel ts params]
    (->> {:token token :channel channel :ts ts}
         (conj params)
         (slack-post "mpim.list"))))

(defn mpim-open
  ([token params]
    (->> {:token token}
         (conj params)
         (slack-post "mpim.open" :group))))

(defn oauth-access
  ([client_id client_secret code params]
    (->> {:client_id client_id :client_secret client_secret :code code}
         (conj params)
         (slack-post "oauth.access"))))

(defn reactions-add
  ([token name params]
    (->> {:token token :name name}
         (conj params)
         (slack-post "reactions.add"))))

(defn reactions-get
  ([token params]
    (->> {:token token}
         (conj params)
         (slack-post "reactions.get"))))

(defn reactions-list
  ([token params]
    (->> {:token token}
         (conj params)
         (slack-post "reactions.list"))))

(defn reactions-remove
  ([token name params]
    (->> {:token token :name name}
         (conj params)
         (slack-post "reactions.remove"))))

(defn rtm-start
  ([token params]
    (->> {:token token}
         (conj params)
         (slack-post "rtm.start"))))

(defn search-all
  ([token query params]
    (->> {:token token :query query}
         (conj params)
         (slack-post "search.all"))))

(defn search-files
  ([token query params]
    (->> {:token token :query query}
         (conj params)
         (slack-post "search.files"))))

(defn search-messages
  ([token query params]
    (->> {:token token :query query}
         (conj params)
         (slack-post "search.messages"))))

(defn stars-add
  ([token params]
    (->> {:token token}
         (conj params)
         (slack-post "stars.add"))))

(defn stars-remove
  ([token params]
    (->> {:token token}
         (conj params)
         (slack-post "stars.remove"))))

(defn team-access-logs
  ([token params]
    (->> {:token token}
         (conj params)
         (slack-post "team.accessLogs" :logins))))

(defn team-info
  ([token params]
    (->> {:token token}
         (conj params)
         (slack-post "team.info" :team))))

(defn users-get-presence
  ([token user params]
    (->> {:token token :user user}
         (conj params)
         (slack-post "users.getPresence"))))

(defn users-info
  ([token params]
    (->> {:token token}
         (conj params)
         (slack-post "users.info" :user))))

(defn users-list
  ([token params]
    (->> {:token token}
         (conj params)
         (slack-post "users.list" :users))))

(defn users-set-active
  ([token params]
    (->> {:token token}
         (conj params)
         (slack-post "users.setActive"))))

(defn users-set-presence
  ([token presence params]
    (->> {:token token :presence presence}
         (conj params)
         (slack-post "users.setActive"))))
