(ns chard.sqs
  (:require [fink-nottle.sqs :as sqs]
            [fink-nottle.sns :as sns]))

(defn get-or-create-dl-queue!! [creds queue-name]
  (let [dl-queue-url (sqs/create-queue!! creds (str queue-name "-dl"))]
    (sqs/queue-arn!! creds dl-queue-url)))

(defn add-redrive-policy [attributes creds queue-name max-receive-count]
  (assoc
    attributes
    :redrive-policy
    {:max-receive-count max-receive-count
     :dead-letter-target-arn (get-or-create-dl-queue!! creds queue-name)}))

(defn subscribe-to-topics!! [creds queue-url topic-arns]
  (let [queue-arn (sqs/queue-arn!! creds queue-url)]
    (sqs/set-queue-attribute!!
      creds
      queue-url
      :policy
      {:version "2012-10-17"
       :statement [{
                    :sid "SnsAllow"
                    :effect "Allow"
                    :principal "*"
                    :action "sqs:SendMessage"
                    :resource queue-arn
                    :condition {
                                "ForAnyValue:StringEquals" {
                                                            :aws/source-arn topic-arns}}}]})
    (doseq [topic-arn topic-arns]
      (let [subscription-arn (sns/subscribe!! creds topic-arn :sqs queue-arn)]
        (sns/set-subscription-attribute!! creds subscription-arn :raw-message-delivery "true"))
      )))

(defn get-or-create-queue!!
  ([creds queue-name]
    (get-or-create-queue!! creds queue-name {}))
  ([creds queue-name opts]
   (let [redrive (or (:redrive opts) true)
         redrive-max-retries (if (number? redrive) redrive 5)
         subscriptions (or (:subscriptions opts) [])
         raw-attributes (dissoc opts :redrive :subscriptions)
         attributes (if (true? redrive)
                       (add-redrive-policy raw-attributes creds queue-name redrive-max-retries)
                       raw-attributes)
         queue-url (sqs/create-queue!! creds queue-name {:attrs attributes})
         topic-arns (map #(sns/create-topic!! creds %) subscriptions)]
     (when-not (empty? subscriptions)
       (subscribe-to-topics!! creds queue-url topic-arns))
      queue-url)))
