package com.socialquantum.battleship.gameapi.types.connector;


import com.socialquantum.battleship.gameapi.types.base.GameId;
import com.socialquantum.battleship.gameapi.types.base.SnapshotId;
import com.socialquantum.battleship.gameapi.types.base.UserToken;
import com.socialquantum.battleship.gameapi.types.exceptions.OperationFailed;
import com.socialquantum.battleship.gameapi.types.request.placeships.ShipsPlacementRequest;
import com.socialquantum.battleship.gameapi.types.response.*;

import java.io.Closeable;

/**
 * !NB! Возвращаемый тип каждого метода должен быть GeneralOperationResponse + описание того что может быть возвращено
 * Основоной интерфейс взаимодействия с игрой.
 * Большинство методов интерфейса предполагают возврат обобщенных данных
 * Created 27/06/17 16:40
 *
 * @author Vladimir Bogodukhov
 * @since version 1
 * {@link com.socialquantum.battleship.gameapi.types.connector.GameConnector}
 **/
public interface GameConnector extends Closeable {

    /**
     * Создать игру.
     *
     * @param player1Token токен первого игрока
     * @param player2Token токен второго игрока
     * @return Результат выполнения операции
     * @throws OperationFailed в случае неуспешной операции
     * @since version 1
     */
    CreateGame createGame(UserToken player1Token, UserToken player2Token);

    /**
     * Начать игру.
     *
     * @param gameId идентификатор игры
     * @return Объект результата выполненной команды
     * @throws OperationFailed в случае неуспешной операции
     * @since version 1
     */
    StartGame startGame(GameId gameId);

    /**
     * Удалить игру.
     *
     * @param gameId идентфиикатор игры
     * @return Результат выполненной команды
     * @throws OperationFailed в случае неуспешной операции
     * @since version 1
     */
    DeleteGame deleteGame(GameId gameId);


    /**
     * Получить список всех игр для игрока
     *
     * @return Результат выполненной команды
     * @throws OperationFailed в случае неуспешной операции
     * @since version 1
     */
    GameList gameList();

    /**
     * Получить текущий статус игры.
     *
     * @param gameId идентфиикатор игры.
     * @return Объект результата выполненной команды
     * @throws OperationFailed в случае неуспешной операции
     * @since version 1
     */
    GameStatus gameStatus(GameId gameId);

    /**
     * Расположить корабли.
     *
     * @param shipsPlacement - объект расположения данных о кораблях
     * @return Объект результата выполненной команды
     * @throws OperationFailed в случае неуспешной операции
     * @since version 1
     */
    PlaceShips placeShips(ShipsPlacementRequest shipsPlacement);


    /**
     * Осуществить выстрел.
     *
     * @param gameId - идентфикатор игры
     * @param x      - координата x
     * @param y      - координата y
     * @return Объект результата выполненной команды
     * @throws OperationFailed в случае неуспешной операции
     * @since version 1
     */
    Shoot shoot(GameId gameId, int x, int y);

    /**
     * Сохранить состояние игры.
     *
     * @param gameId идентификатор игры
     * @return Объект результата выполненной команды
     * @throws OperationFailed в случае неуспешной операции
     * @since version 1
     */
    SaveSnapshot saveSnapshoot(GameId gameId);

    /**
     * Восстановить состояние игры.
     *
     * @param snapshotId идентфиикатор состояния.
     * @return Объект результата выполненной команды
     * @throws OperationFailed в случае неуспешной операции
     * @since version 1
     */
    RestoreSnapshot restoreSnapshot(SnapshotId snapshotId);


    /**
     * Получить список всех сохраненных состояний
     *
     * @return Объект результата выполненной команды
     * @throws OperationFailed в случае неуспешной операции
     * @since version 1
     */
    SnapshotList snapshotList();


    /**
     * Удалить сохраненное состояние.
     *
     * @param snapshotId идентфиикатор сохраенных данных
     * @return Объект результата выполненной команды
     * @throws OperationFailed в случае неуспешной операции
     * @since version 1
     */
    DeleteSnapshot deleteSnapshot(SnapshotId snapshotId);

}
