package com.socialquantum.battleship.gameapi.internal.connector;

import com.socialquantum.battleship.gameapi.internal.dataobjects.creategame.CreateGameRequestData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.creategame.CreateGameResponseData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.defaultresponse.DefaultResponse;
import com.socialquantum.battleship.gameapi.internal.dataobjects.deletegame.DeleteGameRequestData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.deletegame.DeleteGameResponseData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.deletesnapshot.DeleteSnapshotRequestData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.deletesnapshot.DeleteSnapshotResponseData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.gamelist.GameListRequestData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.gamelist.GameListResponseData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.gamestatus.GameStatusRequestData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.gamestatus.GameStatusResponseData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.move.ShootRequestData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.move.ShootResponseData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.placeships.PlaceShipsResponseData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.restoresnapshot.RestoreSnapshotRequestData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.restoresnapshot.RestoreSnapshotResponseData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.savesnapshot.SaveSnapshotRequestData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.savesnapshot.SaveSnapshotResponseData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.snapshotlist.SnapshotListRequestData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.snapshotlist.SnapshotListResponseData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.startgame.StartGameRequestData;
import com.socialquantum.battleship.gameapi.internal.dataobjects.startgame.StartGameResponseData;
import com.socialquantum.battleship.gameapi.internal.serializationutils.SerializationDeserializationFactory;
import com.socialquantum.battleship.gameapi.internal.types.OperationResult;
import com.socialquantum.battleship.gameapi.types.base.ExecutionStatus;
import com.socialquantum.battleship.gameapi.types.base.GameId;
import com.socialquantum.battleship.gameapi.types.base.SnapshotId;
import com.socialquantum.battleship.gameapi.types.base.UserToken;
import com.socialquantum.battleship.gameapi.types.connector.GameConnector;
import com.socialquantum.battleship.gameapi.types.exceptions.OperationException;
import com.socialquantum.battleship.gameapi.types.request.placeships.ShipsPlacementRequest;
import com.socialquantum.battleship.gameapi.types.response.*;
import com.socialquantum.battleship.gameapi.internal.utils.GameUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.log4j.Logger;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.Objects;
import java.util.function.Function;

import static com.socialquantum.battleship.gameapi.internal.utils.GameUtils.wrapException;
import static com.socialquantum.battleship.gameapi.internal.utils.GameUtils.wrapExceptionWoResult;

/**
 * Created 28/06/17 14:01
 *
 * @author Vladimir Bogodukhov
 *         Информация по используемым библиотекам
 * @see <a href="https://en.wikipedia.org/wiki/Gson">Описание в wikipedia Google-Gson</a>
 * @see <a href="https://github.com/google/gson/blob/master/UserGuide.md">Описание от производителя.</a>
 **/
public class GameConnectionVer1 implements GameConnector {

    private interface ResultData<SuccessType, ErrorType> {
        SuccessType success();

        ErrorType error();

        ExecutionStatus status();
    }

    private static class DefaultResultData<SuccessType, ErrorType> implements ResultData<SuccessType, ErrorType> {
        private final ExecutionStatus executionStatus;
        private final SuccessType success;
        private final ErrorType error;

        private DefaultResultData(ExecutionStatus executionStatus, SuccessType success, ErrorType error) {
            this.executionStatus = Objects.requireNonNull(executionStatus);
            switch (executionStatus) {
                case OK:
                    Objects.requireNonNull(success);
                    break;

                case ERROR:
                    Objects.requireNonNull(error);
                    break;

                default:
                    throw new IllegalStateException();
            }

            this.success = success;
            this.error = error;
        }

        @Override
        public SuccessType success() {
            return success;
        }

        @Override
        public ErrorType error() {
            return error;
        }

        @Override
        public ExecutionStatus status() {
            return executionStatus;
        }
    }

    private static final Logger LOG = Logger.getLogger(GameConnectionVer1.class);
    private static final String API_TOKEN_HEADER = "apiToken";
    private static final Charset DOCUMENT_CHARSET = Charset.forName("utf-8");

    /**
     * Адрес где находятся все функции игры.
     */
    private static final String GAME_URL_FUNCTION = "/game";


    private static class CustomDataResponseHandler<SuccessType, ErrorType> implements
            ResponseHandler<OperationResult<SuccessType, ErrorType>> {

        private final Function<String, ResultData<SuccessType, ErrorType>> handlerFunction;

        private CustomDataResponseHandler(Function<String, ResultData<SuccessType, ErrorType>> handlerFunction) {
            this.handlerFunction = Objects.requireNonNull(handlerFunction);
        }

        @Override
        public OperationResult<SuccessType, ErrorType> handleResponse(HttpResponse response) throws IOException {
            try (InputStream stream = response.getEntity().getContent()) {
                String dataAsString = loadString(stream);
                int httpResponseCode = response.getStatusLine().getStatusCode();
                LOG.info("\r\nresponse status " + httpResponseCode + "\r\nresponse body \r\n " + dataAsString);

                ResultData<SuccessType, ErrorType> result = handlerFunction.apply(dataAsString);
                return GameUtils.result(httpResponseCode, recodeStatus(result.status()), result.success(), result.error());
            }
        }
    }

    /**
     * Обработчик запроса для большинства случаев.
     */
    private static class GeneralResponseHandler<SuccessType, ErrorType> implements
            ResponseHandler<OperationResult<SuccessType, ErrorType>> {

        private final Class<SuccessType> successResultDataClass;
        private final Class<ErrorType> errorResultDataClass;

        private GeneralResponseHandler(
                Class<SuccessType> successResultDataClass,
                Class<ErrorType> errorResultDataClass) {

            this.successResultDataClass = Objects.requireNonNull(successResultDataClass);
            this.errorResultDataClass = Objects.requireNonNull(errorResultDataClass);
        }

        @Override
        public OperationResult<SuccessType, ErrorType> handleResponse(HttpResponse response) throws IOException {
            try (InputStream stream = response.getEntity().getContent()) {
                String dataAsString = loadString(stream);
                int httpResponseCode = response.getStatusLine().getStatusCode();
                LOG.info("\r\nresponse status " + httpResponseCode + "\r\nresponse body \r\n " + dataAsString);

                switch (SERIALIZATION_DESERIALIZATION_FACTORY.loadObject(dataAsString, DefaultResponse.class).status()) {
                    case ERROR: {
                        return GameUtils.result(httpResponseCode, OperationResult.Result.ERROR, null,
                                SERIALIZATION_DESERIALIZATION_FACTORY.loadObject(dataAsString, errorResultDataClass));
                    }

                    case OK: {
                        return GameUtils.result(httpResponseCode, OperationResult.Result.SUCCESS,
                                SERIALIZATION_DESERIALIZATION_FACTORY.loadObject(dataAsString, successResultDataClass), null);
                    }
                    default:
                        throw new IllegalStateException();
                }
            }
        }
    }


    private static String loadString(InputStream stream) throws IOException {

        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        {
            byte[] data = new byte[1024];
            int readBytesCount;
            while ((readBytesCount = stream.read(data)) > 0) {
                byteArrayOutputStream.write(data, 0, readBytesCount);
            }
        }

        return byteArrayOutputStream.toString(DOCUMENT_CHARSET.name());
    }


    private final CloseableHttpClient httpclient = HttpClients.createDefault();
    private final String gamefunctions;
    private final UserToken playerTokenId;
    private static final SerializationDeserializationFactory SERIALIZATION_DESERIALIZATION_FACTORY =
            new SerializationDeserializationFactory();


    public GameConnectionVer1(String host, UserToken playerTokenId) {
        wrapExceptionWoResult(IllegalStateException.class, IllegalStateException::new, () -> new URL(host));
        this.gamefunctions = Objects.requireNonNull(host) + GAME_URL_FUNCTION;
        this.playerTokenId = Objects.requireNonNull(playerTokenId);
    }

    /**
     * Закрытие коннектора, освобождение ресурсов.
     */
    @Override
    public void close() throws IOException {
        wrapExceptionWoResult(IllegalStateException.class, IllegalStateException::new, httpclient::close);
    }

    /* ------------------------------------------------------------------------------------------------------------------ */

    @Override
    public OperationResponse createGame(UserToken player1Token, UserToken player2Token) {
        LOG.info("createGame");
        CreateGameRequestData createGameDataObject = new CreateGameRequestData().
                setUsers(player1Token, player2Token);

        OperationResult<CreateGameResponseData, DefaultResponse> post = post(createGameDataObject,
                CreateGameResponseData.class, DefaultResponse.class);

        switch (post.result()) {
            case SUCCESS: {
                return post.successData();
            }

            case ERROR: {
                return post.errorData();
            }
            default:
                throw new IllegalStateException("unknown result " + post.result());
        }
    }

    @Override
    public OperationResponse startGame(GameId gameId) {
        LOG.info("startGame");
        StartGameRequestData createGameDataObject = new StartGameRequestData().setGameId(gameId);

        OperationResult<StartGameResponseData, DefaultResponse> post =
                post(createGameDataObject, StartGameResponseData.class, DefaultResponse.class);

        switch (post.result()) {
            case SUCCESS: {
                return post.successData();
            }

            case ERROR: {
                return post.errorData();
            }
            default:
                throw new IllegalStateException("unknown result " + post.result());
        }
    }

    @Override
    public OperationResponse deleteGame(GameId gameId) {
        LOG.info("deleteGame");
        Objects.requireNonNull(gameId);
        DeleteGameRequestData deleteGame = new DeleteGameRequestData().setGameId(gameId);
        OperationResult<DeleteGameResponseData, DefaultResponse> post =
                post(deleteGame, DeleteGameResponseData.class, DefaultResponse.class);

        switch (post.result()) {
            case SUCCESS: {
                return post.successData();
            }

            case ERROR: {
                return post.errorData();
            }
            default:
                throw new IllegalStateException("unknown result " + post.result());
        }
    }

    @Override
    public OperationResponse gameList() {
        LOG.info("gameList");
        GameListRequestData deleteGame = new GameListRequestData();
        OperationResult<GameListResponseData, DefaultResponse> post =
                post(deleteGame, GameListResponseData.class, DefaultResponse.class);

        switch (post.result()) {
            case SUCCESS: {
                return post.successData();
            }

            case ERROR: {
                return post.errorData();
            }
            default:
                throw new IllegalStateException("unknown result " + post.result());
        }
    }

    @Override
    public OperationResponse gameStatus(GameId gameId) {
        LOG.info("gameStatus");
        GameStatusRequestData statusRequestData = new GameStatusRequestData().setGameId(gameId);

        OperationResult<GameStatusResponseData, DefaultResponse> post =
                post(statusRequestData, GameStatusResponseData.class, DefaultResponse.class);

        switch (post.result()) {
            case SUCCESS: {
                return post.successData();
            }

            case ERROR: {
                return post.errorData();
            }
            default:
                throw new IllegalStateException("unknown result " + post.result());
        }
    }

    @Override
    public OperationResponse placeShips(ShipsPlacementRequest shipsPlacement) {
        LOG.info("placeShips");
        Objects.requireNonNull(shipsPlacement);
        OperationResult<PlaceShipsResponseData, DefaultResponse> post = post(shipsPlacement,
                PlaceShipsResponseData.class, DefaultResponse.class);

        switch (post.result()) {
            case SUCCESS: {
                return post.successData();
            }

            case ERROR: {
                return post.errorData();
            }

            default:
                throw new IllegalStateException("unknown result " + post.result());
        }
    }

    @Override
    public OperationResponse shoot(GameId gameId, int x, int y) {
        LOG.info("shoot");
        Objects.requireNonNull(gameId);
        ShootRequestData makeMoveRequestData = new ShootRequestData().setCoords(x, y).setGameId(gameId);
        OperationResult<ShootResponseData, DefaultResponse> post = post(makeMoveRequestData,
                ShootResponseData.class, DefaultResponse.class);
        switch (post.result()) {
            case SUCCESS:
                return post.successData();
            case ERROR: {
                return post.errorData();
            }

            default:
                throw new IllegalStateException("unknown result " + post.result());
        }
    }


    @Override
    public OperationResponse saveSnapshoot(GameId gameId) {
        LOG.info("saveSnapshoot");
        SaveSnapshotRequestData makeMoveRequestData = new SaveSnapshotRequestData().setGameId(gameId);
        OperationResult<SaveSnapshotResponseData, DefaultResponse> post = post(makeMoveRequestData,
                SaveSnapshotResponseData.class, DefaultResponse.class);
        switch (post.result()) {
            case SUCCESS:
                return post.successData();
            case ERROR: {
                return post.errorData();
            }
            default:
                throw new IllegalStateException("unknown result " + post.result());
        }
    }

    @Override
    public OperationResponse restoreSnapshot(SnapshotId snapshotId) {
        LOG.info("restoreSnapshot");
        RestoreSnapshotRequestData requestData = new RestoreSnapshotRequestData().setSnapshotId(snapshotId);
        OperationResult<RestoreSnapshotResponseData, DefaultResponse> post = post(requestData,
                RestoreSnapshotResponseData.class, DefaultResponse.class);
        switch (post.result()) {
            case SUCCESS:
                return post.successData();
            case ERROR: {
                return post.errorData();
            }
            default:
                throw new IllegalStateException("unknown result " + post.result());
        }
    }

    @Override
    public OperationResponse snapshotList() {
        LOG.info("snapshotList");
        SnapshotListRequestData makeMoveRequestData = new SnapshotListRequestData();
        OperationResult<SnapshotListResponseData, DefaultResponse> post = post(makeMoveRequestData,
                SnapshotListResponseData.class, DefaultResponse.class);
        switch (post.result()) {
            case SUCCESS:
                return post.successData();
            case ERROR: {
                return post.errorData();
            }
            default:
                throw new IllegalStateException("unknown result " + post.result());
        }
    }


    @Override
    public OperationResponse deleteSnapshot(SnapshotId snapshotId) {
        LOG.info("deleteSnapshot ");
        DeleteSnapshotRequestData requestData = new DeleteSnapshotRequestData().setSnapshotId(snapshotId);
        OperationResult<DeleteSnapshotResponseData, DefaultResponse> post = post(requestData,
                DeleteSnapshotResponseData.class, DefaultResponse.class);

        switch (post.result()) {
            case SUCCESS:
                return post.successData();
            case ERROR: {
                return post.errorData();
            }
            default:
                throw new IllegalStateException("unknown result " + post.result());
        }
    }










/*  ------------------------------------------------ privates  -----------------------------------------------------  */

    /**
     * @param requestData         Данные запроса.
     * @param successResponseType Тип данных успешного ответа.
     * @param errorResponseType   Тип данных неуспешного ответа.
     * @throws OperationException при любой ошибке взаимодействия с сервером игры.
     */
    private <RQ, SuccessResponse, ErrorResponse> OperationResult<SuccessResponse, ErrorResponse> post(
            RQ requestData,
            Class<SuccessResponse> successResponseType,
            Class<ErrorResponse> errorResponseType) {

        Objects.requireNonNull(requestData);
        Objects.requireNonNull(successResponseType);
        Objects.requireNonNull(errorResponseType);

        HttpPost request = new HttpPost(gamefunctions);
        request.addHeader(API_TOKEN_HEADER, playerTokenId.value());

        String requestString = SERIALIZATION_DESERIALIZATION_FACTORY.saveObject(requestData);

        LOG.info("request \r\n" + requestString);
        request.setEntity(new StringEntity(requestString, DOCUMENT_CHARSET));

        return wrapException(OperationException.class, OperationException::new,
                () -> httpclient.execute(request,
                        new GeneralResponseHandler<>(successResponseType, errorResponseType)));
    }

    /**
     * @param requestData Данные запроса.
     * @param handler     Функционал обработки возвращаемых данных
     * @throws OperationException при любой ошибке взаимодействия с сервером игры.
     */
    private <RQ, SuccessResponse, ErrorResponse> OperationResult<SuccessResponse, ErrorResponse> post(
            RQ requestData, Function<String, ResultData<SuccessResponse, ErrorResponse>> handler) {

        Objects.requireNonNull(requestData);
        Objects.requireNonNull(handler);

        HttpPost request = new HttpPost(gamefunctions);
        request.addHeader(API_TOKEN_HEADER, playerTokenId.value());

        String requestString = SERIALIZATION_DESERIALIZATION_FACTORY.saveObject(requestData);

        LOG.info("request \r\n" + requestString);
        request.setEntity(new StringEntity(requestString, DOCUMENT_CHARSET));

        return wrapException(OperationException.class, OperationException::new,
                () -> httpclient.execute(request,
                        new CustomDataResponseHandler<>(handler)));
    }

    private static OperationResult.Result recodeStatus(ExecutionStatus executionStatus) {
        switch (executionStatus) {
            case OK:
                return OperationResult.Result.SUCCESS;
            case ERROR:
                return OperationResult.Result.ERROR;
            default:
                throw new IllegalStateException();
        }
    }


}
