package com.socialquantum.battleship.gameapi.types.connector;


import com.socialquantum.battleship.gameapi.types.base.SnapshotId;
import com.socialquantum.battleship.gameapi.types.base.UserToken;
import com.socialquantum.battleship.gameapi.types.base.GameId;
import com.socialquantum.battleship.gameapi.types.exceptions.UnexpectedOperationResult;
import com.socialquantum.battleship.gameapi.types.request.placeships.ShipsPlacementRequest;
import com.socialquantum.battleship.gameapi.types.response.*;

import java.io.Closeable;

/**
 * Основоной интерфейс взаимодействия с игрой.
 * Большинство методов интерфейса предполагают возврат обобщенных данных результата выполнения команды взаимодействия.
 * Конкретный тип возвращаемого объекта для случаем успеха и не успеха операции указаны в описании каждого метода.
 * Приведение к желаемому типу можно делать как стандартными средствами языка
 * (ResultType) result, так и с использованием метода {@link #cast(Object, Class)} определенного в этом интерфейсе.
 * Created 27/06/17 16:40
 *
 * @author Vladimir Bogodukhov
 * @since version 1
 * {@link com.socialquantum.battleship.gameapi.types.connector.GameConnector}
 **/
public interface GameConnector extends Closeable {

    /**
     * Создать игру.
     * Типы возвращаемые в случае успеха и не успеха операции соответственно:
     * {@link CreateGame} {@link OperationError}
     *
     * @param player1Token токен первого игрока
     * @param player2Token токен второго игрока
     * @return Результат выполнения операции
     * @since version 1
     */
    OperationResponse createGame(UserToken player1Token, UserToken player2Token);

    /**
     * Начать игру.
     * Типы возвращаемые в случае успеха и не успеха операции соответственно:
     * {@link StartGame} {@link OperationError}
     *
     * @param gameId идентификатор игры
     * @return Объект результата выполненной команды
     * @since version 1
     */
    OperationResponse startGame(GameId gameId);

    /**
     * Удалить игру.
     * Типы возвращаемые в случае успеха и не успеха операции соответственно:
     * {@link DeleteGame} {@link OperationError}
     *
     * @param gameId идентфиикатор игры
     * @return Результат выполненной команды
     * @since version 1
     */
    OperationResponse deleteGame(GameId gameId);


    /**
     * Получить список всех игр для игрока
     * Типы возвращаемые в случае успеха и не успеха операции соответственно:
     * {@link GameList} {@link OperationError}
     *
     * @return Результат выполненной команды
     * @since version 1
     */
    OperationResponse gameList();

    /**
     * Получить текущий статус игры.
     * <p>
     * Типы возвращаемые в случае успеха и не успеха операции соответственно:
     * {@link GameStatus} {@link OperationError}
     *
     * @param gameId идентфиикатор игры.
     * @return Объект результата выполненной команды
     * @since version 1
     */
    OperationResponse gameStatus(GameId gameId);

    /**
     * Расположить корабли.
     * Типы возвращаемые в случае успеха и не успеха операции соответственно:
     * {@link PlaceShips} {@link OperationError}
     *
     * @param shipsPlacement - объект расположения данных о кораблях
     * @return Объект результата выполненной команды
     * @since version 1
     */
    OperationResponse placeShips(ShipsPlacementRequest shipsPlacement);


    /**
     * Выстрел.
     * Типы возвращаемые в случае успеха и не успеха операции соответственно:
     * {@link Shoot} {@link OperationError}
     *
     * @param gameId - идентфикатор игры
     * @param x      - координата x
     * @param y      - координата y
     * @return Объект результата выполненной команды
     * @since version 1
     */
    OperationResponse shoot(GameId gameId, int x, int y);

    /**
     * Сохранить состояние игры.
     * Типы возвращаемые в случае успеха и не успеха операции соответственно:
     * {@link SaveSnapshot} {@link OperationError}
     *
     * @param gameId идентификатор игры
     * @return Объект результата выполненной команды
     * @since version 1
     */
    OperationResponse saveSnapshoot(GameId gameId);

    /**
     * Восстановить состояние игры.
     * Типы возвращаемые в случае успеха и не успеха операции соответственно:
     * {@link RestoreSnapshot} {@link OperationError}
     *
     * @param snapshotId идентфиикатор состояния.
     * @return Объект результата выполненной команды
     * @since version 1
     */
    OperationResponse restoreSnapshot(SnapshotId snapshotId);


    /**
     * Получить список всех сохраненных состояний
     * Типы возвращаемые в случае успеха и не успеха операции соответственно:
     * {@link SnapshotList} {@link OperationError}
     *
     * @return Объект результата выполненной команды
     * @since version 1
     */
    OperationResponse snapshotList();


    /**
     * Удалить сохраненное состояние.
     * Типы возвращаемые в случае успеха и не успеха операции соответственно:
     * {@link DeleteSnapshot} {@link OperationError}
     *
     * @param snapshotId идентфиикатор сохраенных данных
     * @return Объект результата выполненной команды
     * @since version 1
     */
    OperationResponse deleteSnapshot(SnapshotId snapshotId);


    /**
     * Вспомогательный метод приведения объекта к требуемому типу
     * По сути является, компактной записью для (SomeType) object, при этом если объект не является типом,
     * к которому производится приведение будет выброшено исключение {@link UnexpectedOperationResult}, а не {@link ClassCastException}
     * как это происходит в случае приведения средствами языка.
     *
     * @param o     объект
     * @param clazz приводимый тип
     * @param <T>   приводимый тип
     * @return объект, приведенный к типу
     * @throws UnexpectedOperationResult в случае попытки приведения к неверному типу
     */
    static <T> T cast(Object o, Class<T> clazz) {
        if (o == null) {
            //noinspection unchecked
            return (T) o;
        }

        if (!clazz.isAssignableFrom(o.getClass())) {
            throw new UnexpectedOperationResult("expected type " + clazz.getName() + " cannot be applied to object " + o);
        }
        return clazz.cast(o);
    }

}
