package com.socialquantum.battleship.gameapi.internal.serializationutils;

import com.google.gson.*;
import com.socialquantum.battleship.gameapi.internal.dataobjects.gamestatus.GameStateDataStatus;
import com.socialquantum.battleship.gameapi.types.base.GameId;
import com.socialquantum.battleship.gameapi.types.base.UserToken;
import com.socialquantum.battleship.gameapi.types.base.SnapshotId;

import java.lang.reflect.Type;
import java.util.Objects;

import static com.socialquantum.battleship.gameapi.utils.TransformUtils.gameId;
import static com.socialquantum.battleship.gameapi.utils.TransformUtils.snapshotId;
import static com.socialquantum.battleship.gameapi.utils.TransformUtils.userToken;

/**
 * Фабрика загрузки и сохранения объектов.
 * Created 29/06/17 13:33
 *
 * @author Vladimir Bogodukhov
 **/
public class SerializationDeserializationFactory {

    private static class GameStateDataStatusAdapter
            implements JsonSerializer<GameStateDataStatus>, JsonDeserializer<GameStateDataStatus> {

        @Override
        public JsonElement serialize(GameStateDataStatus src, Type typeOfSrc, JsonSerializationContext context) {
            if (src == null) return null;
            return new JsonPrimitive(src.ident());
        }

        @Override
        public GameStateDataStatus deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
                throws JsonParseException {
            return GameStateDataStatus.elementByName(json.getAsString());
        }
    }

    private static class GameIdAdapter implements JsonSerializer<GameId>, JsonDeserializer<GameId> {
        @Override
        public GameId deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context) throws JsonParseException {
            return gameId(json.getAsString());
        }

        @Override
        public JsonElement serialize(GameId src, Type typeOfSrc, JsonSerializationContext context) {
            return src == null ? null : new JsonPrimitive(src.value());
        }
    }

    private static class UserTokenAdapter implements JsonSerializer<UserToken>, JsonDeserializer<UserToken> {
        @Override
        public UserToken deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context) throws JsonParseException {
            return userToken(json.getAsString());
        }

        @Override
        public JsonElement serialize(UserToken src, Type typeOfSrc, JsonSerializationContext context) {
            return src == null ? null : new JsonPrimitive(src.value());
        }
    }

    private static class SnapshotIdAdapter implements JsonSerializer<SnapshotId>, JsonDeserializer<SnapshotId> {
        @Override
        public SnapshotId deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context) throws JsonParseException {
            return snapshotId(json.getAsString());
        }

        @Override
        public JsonElement serialize(SnapshotId src, Type typeOfSrc, JsonSerializationContext context) {
            return src == null ? null : new JsonPrimitive(src.value());
        }
    }


    private static final GsonBuilder GSON_BUILDER = configBuilder();


    /**
     * Загрузить объект типа из строки
     *
     * @param data  данные в строке
     * @param clazz тип к которому приводятся данные
     * @param <T>   тип к которому приводятся данные
     * @return объект типа
     */
    public <T> T loadObject(String data, Class<T> clazz) {
        return GSON_BUILDER.create().fromJson(data, clazz);
    }


    /**
     * Сохранить данные объекта в строку.
     *
     * @param dataObject - объект данных
     * @return строка с данными
     */
    public String saveObject(Object dataObject) {
        Objects.requireNonNull(dataObject);
        return GSON_BUILDER.create().toJson(dataObject);
    }

    /*  ------------------------------------------------------------------------------------------------------------  */

    private static GsonBuilder configBuilder() {
        GsonBuilder gsonBuilder = new GsonBuilder();
        gsonBuilder.registerTypeAdapter(GameStateDataStatus.class, new GameStateDataStatusAdapter());
        gsonBuilder.registerTypeAdapter(GameId.class, new GameIdAdapter());
        gsonBuilder.registerTypeAdapter(UserToken.class, new UserTokenAdapter());
        gsonBuilder.registerTypeAdapter(SnapshotId.class, new SnapshotIdAdapter());
        return gsonBuilder;
    }

}
