(ns smx.eventstore.search.system
  (:require [smx.eventstore.search.nrepl :as nr]
            [smx.eventstore.search.model :as model]
            [smx.eventstore.search.state :as state]
            [smx.eventstore.search.planner :as pl]
            [smx.eventstore.search.executor :as exec]
            [smx.eventstore.search.search :as srh]
            [smx.eventstore.search.searches :as searches]
            [smx.eventstore.config :as conf]
            [smx.eventstore.cassandra.core :as cassandra]
            [smx.eventstore.cassandra.event :as cassandra.event]
            [smx.netty.application :as app]
            [smx.netty.http.server :as http]
            [smx.netty.http.route :as r]
            [smx.netty.http.admin.server :as admin]
            [clojure.tools.logging :refer [info error warn debug]]
            [metrics.core :as m]
            [clojure.java.io :as io]
            [clojure.tools.logging :as log]))

(defn ->application-config [search-opts]
  (System/setProperty "java.net.preferIPv4Stack" "true")
  (let [registry (m/new-registry)
        server-cfg (conf/gather :search-server)]
    (info "Default search options are:" search-opts)
    (info "Server config is " server-cfg)
    (info "(:search-ttl-secs search-opts) is " (:search-ttl-secs search-opts))
    (app/->application registry (admin/->server :admin-server)
      ;; configure the search server
      (http/->server
        :search-server
        (app/->component :nrepl (nr/new-repl (:port (conf/gather :nrepl))))
        (app/->component :cluster (cassandra/->cluster (:cluster (conf/gather :cassandra))))
        (app/->component :cassandra (cassandra/->connection (:keyspace (conf/gather :cassandra)) cassandra.event/queries))
        (app/->component :model (model/new-model (conf/parse-edn (io/resource "model/archiving-model.edn"))))
        (app/->component :state (state/new-state))
        (app/->component :planner (pl/new-planner search-opts))
        (app/->component :searches (searches/new-searches (:search-ttl-secs search-opts)))
        (app/->component :executor (exec/new-executor (:search-ttl-secs search-opts) (:page-size search-opts)))

        ;; routes
        (r/->route :search-archive :post "/mail/search" (srh/search-handler server-cfg))
        (r/->route :search-archive-next :post "/mail/search/:search-id/next" (srh/next-results-handler server-cfg)))

      ;; dependencies that are not resolved by the DSL automatically must be set manually
      (app/with-extra-deps
        {:cassandra           [:cluster]
         :state               [:cassandra :model]
         :planner             [:model :state]
         :searches            [:cassandra]
         :executor            [:searches]
         :search-archive      [:planner :searches :executor]
         :search-archive-next [:searches]}))))

(defn start! []
  (log/info "Starting search system")
  (.setName (Thread/currentThread) "Eventstore-Search")
  (let [config (->application-config (conf/gather :search-options))]
    (app/start! config)))

(defn stop! []
  (log/info "Stopping search system")
  (app/stop!))

(Thread/setDefaultUncaughtExceptionHandler
  (reify Thread$UncaughtExceptionHandler
    (uncaughtException [_ thread ex]
      (.printStackTrace ex)                                 ;log goes missing sometimes?
      (log/errorf ex "Uncaught exception on %" (.getName thread)))))


