(ns smx.eventstore.search.searches
  (:require [clojure.core.async :as async]
            [clojure.core.cache :as cache]
            [clojure.tools.logging :as log]
            [smx.eventstore.search.log :as slog]
            [smx.eventstore.cassandra.core :as cassandra]
            [com.stuartsierra.component :as component]
            [schema.core :as s])
  (:import [clojure.core.async.impl.protocols Channel]
           [com.datastax.driver.core Session]
           [smx.eventstore.cassandra.core Connection]
           [clojure.lang Keyword]))

;;;;;;;;;;;;;
;;; Schema

(s/defrecord Searches [cassandra :- Connection
                       search-ttl-secs :- Long
                       search-cache
                       last-ttl-check
                       ctrl-buffer-size :- Long
                       error-buffer-size :- Long]
  component/Lifecycle
  (start [this]
    (log/info "Starting searches component")
    this)
  (stop [this]
    (log/info "Stopping searches component")
    this))

(s/defrecord Context [search-id :- Long
                      search :- s/Any
                      ctrl-chan :- Channel
                      err-chan :- Channel
                      session :- (s/maybe Session)
                      peek-chan :- (s/maybe Channel)
                      results-chan :- (s/maybe Channel)
                      peek :- (s/maybe {Keyword s/Any})])

;;;;;;;
;;; API

(s/defn new-searches [search-ttl-secs :- Long]
  (map->Searches
    {:search-ttl-secs   search-ttl-secs
     :error-buffer-size 10
     :ctrl-buffer-size  1
     :last-ttl-check    (atom (System/currentTimeMillis))
     :search-cache      (atom (cache/ttl-cache-factory {} :ttl (* search-ttl-secs 1000)))}))

(s/defn ->context [searches :- Searches
                   search-id :- Long]
  (map->Context
    {:search-id search-id
     :session   (-> searches :cassandra :session)
     :ctrl-chan (async/chan (:ctrl-buffer-size searches))
     :err-chan  (async/chan (:error-buffer-size searches))}))

(s/defn cache-context [this :- Searches
                       context :- Context]
  ;triggers ttl check for other entries
  (swap! (:search-cache this) assoc (:search-id context) context))

(s/defn lookup-context :- (s/maybe Context) [this :- Searches search-id :- Long]
  (cache/lookup @(:search-cache this) search-id))

(s/defn remove-context!
  [this :- Searches
   search-id :- Long]
  (slog/debug "Removing context")
  (swap! (:search-cache this) cache/evict search-id))

(s/defn search-ids :- [Long] [this :- Searches]
  ;need to walk to prevent AbstactMethodError cos no iterator, bug in c.c.c
  (vec (keys @(:search-cache this))))