(ns smx.eventstore.search.search
  (:require [smx.eventstore.search.executor :as executor]
            [smx.eventstore.search.planner :as planner]
            [smx.eventstore.search.searches :as searches]
            [smx.eventstore.search.log :as slog]
            [smx.eventstore.event :as event]
            [clojure.core.async :as async]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [clojure.edn :as edn]
            [schema.core :as s])
  (:import (io.netty.util CharsetUtil)
           (java.nio.charset CharsetDecoder)
           [java.net InetAddress]
           [java.sql Timestamp]))

;;;;;;;;;;;;;
;;; Schema

(defrecord SearchHandler
  [planner
   searches
   executor
   decoder
   search-id-gen
   server-cfg
   base-url])

(defrecord NextPageHandler
  [searches
   decoder
   server-cfg
   base-url])

(def munging true)

;todo Search ? rename ns? schema todo
(s/defn ->search
  [^CharsetDecoder decoder req]
  (let [content (str (.decode decoder (:content req)))
        search (edn/read-string content)]
    (if-not search (throw (ex-info "No search found in request" {:user-msg "No search found in request." :uri req :error :validation}))
                   search)))

(def Results {:status                   Long
              :content                  String
              (s/required-key :key)     String
              (s/optional-key :headers) {String s/Any}})

;;;;;;;;;;;;;
;;; Impl

(def host (.getHostName (InetAddress/getLocalHost)))


(defn smx3-munge-req [req]
  req)

(defn smx3-munge-resp [_ resp]
  (mapv
    (fn [rec]
      (cond-> rec
              (:sender-mta-ip rec)
              (assoc :sender-mta-ip (let [ip (:sender-mta-ip rec)]
                                      (if (instance? InetAddress ip) (.getHostAddress ip) ip)))

              (:received rec)
              (assoc :received (Timestamp. (.getMillis (:received rec))))))
    resp))

(defn ->default-headers [search-id]
  {"SMX-Archiving-Search-ID" (str search-id)})

(defn ->next-header [search-id base-url]
  {"SMX-Archiving-Search-ID" (str search-id)
   "Link"                    [(str "<http://" base-url "/mail/search/" search-id "/next" ">; rel=next")]})

(defn ->response
  ([status content]
   (->response status content nil))
  ([status content headers]
   (let [r {:status  status
            :key     "archiving.search/result"
            :content content}]
     (if headers (assoc r :headers headers) r))))

(s/defn return-error [search-id throwable :- Throwable]
  (let [ex-data (ex-data throwable)]
    (log/error throwable search-id "Query failed:")
    (->response (if (= (:error ex-data) :validation) 400 500)
                (or (:user-msg ex-data) "The server couldn't complete the request at this time.")
                (->default-headers search-id))))

;;;;;;;
;;; API

(s/defn execute-search
  "Executes search for a given query. Waits for results or error from executor chans.
  Returns response map with link header for next page of results if hint comes from peek chan"
  [this :- SearchHandler
   request :- s/Any
   search-id :- Long]

  (try
    (let [{:keys [executor planner searches decoder]} this
          search (smx3-munge-req (->search decoder request))
          _ (slog/debug "Search is" search)
          plan (:_plan (planner/create-plan planner search))]
      ;todo can i push all context stuff to executor? put search on it?
      (let [context (searches/->context searches search-id)
            [peek-chan results-chan] (executor/execute-plan executor plan context)
            [r _] (async/alts!! [results-chan (:err-chan context)])]
        (if (instance? Throwable r)
          (return-error search-id r)
          (do (slog/debug "Query results:" r)
              (let [content (pr-str (if munging (smx3-munge-resp search r) r))
                    ;dont care about first peek [do we ever?], subsequent peeks tells us of subsequent pages
                    first-peek (async/<!! peek-chan)
                    second-peek (async/<!! peek-chan)
                    _ (slog/debug "Peek is" second-peek)
                    headers (if second-peek
                              (do
                                (searches/cache-context searches
                                                        (assoc context :peek-chan peek-chan
                                                                       :search search
                                                                       :results-chan results-chan
                                                                       :peek second-peek))
                                (merge (->default-headers search-id) (->next-header search-id (:base-url this))))
                              (->default-headers search-id))]
                (->response 200 content headers))))))
    (catch Exception e (return-error search-id e))))

(s/defn execute-next-results :- Results
  "Fetches next page of results from the executor chan stored in search-id's context or return 410 gone."
  [this :- NextPageHandler
   search-id :- Long]
  (slog/info "Fetching next page of results")
  (try
    (let [{:keys [searches]} this
          context (searches/lookup-context searches search-id)
          [r _] (if context (async/alts!! [(:results-chan context) (:err-chan context)])
                            (slog/info "No context found"))]
      (cond
        (nil? r) (do (slog/info "Search expired")
                     (searches/remove-context! searches search-id)
                     (->response 410 "The search has expired."))
        (instance? Throwable r) (do (slog/error r "Next request failed:")
                                    (->response 500 "The server couldn't complete the request at this time."))
        :else (do (slog/debug "Query results:" r)
                  (let [context (assoc context :peek (async/<!! (:peek-chan context)))]
                    (searches/cache-context searches context)
                    (->response
                      200
                      (pr-str (if munging (smx3-munge-resp (:search context) r) r))
                      (if (:peek context)
                        (do (slog/info "Next page available")
                            (merge (->default-headers search-id) (->next-header search-id (:base-url this))))
                        (->default-headers search-id)))))))
    (catch Exception e (return-error search-id e))))

;;;;;;;;;;;;;;
;;;; Impl



(extend-type SearchHandler
  component/Lifecycle
  (start [this]
    (assoc this :decoder (.newDecoder CharsetUtil/UTF_8)
                :search-id-gen (atom 0)))
  (stop [this]
    this)

  event/Handler
  (process [this {:keys [out] :as ctx}]
    (event/go-while [req ctx] ::execute-search
                    (let [search-id (swap! (:search-id-gen this) inc)
                          response (execute-search this req search-id)]
                      (slog/debug "Response is" response)
                      (async/put! out response)))))

(defn search-handler
  [server-cfg]
  (map->SearchHandler {:server-cfg server-cfg
                       :base-url   (str host ":" (get-in server-cfg [:service :port]))}))


(extend-type NextPageHandler
  component/Lifecycle
  (start [this]
    (assoc this :decoder (.newDecoder CharsetUtil/UTF_8)))
  (stop [this]
    this)

  event/Handler
  (process [this {:keys [out] :as ctx}]
    (event/go-while [req ctx] ::execute-next-results
                    (let [search-id (Long/parseLong (get-in req [:path-params :search-id]))
                          response (execute-next-results this search-id)]
                      (slog/debug "Response is " response)
                      (async/put! out response)))))

(defn next-results-handler
  [server-cfg]
  (map->NextPageHandler {:server-cfg server-cfg
                         :base-url   (str host ":" (get-in server-cfg [:service :port]))}))

