(ns smx.eventstore.search.model
  (:require [schema.core :as s]
            [clojure.edn :as edn]
            [com.stuartsierra.component :as component]
            [clojure.tools.logging :refer [info error debug]]
            [clojure.java.io :as io]
            [clojure.set :as set]
            [clojure.string :as str])
  (:import [clojure.lang Keyword]))

(def ^:private default-types-jarpath "defaults/types.edn")

(def Cardinality (s/enum :very-low :low :medium :high :very-high))
(def Selectivity (s/enum :very-low :low :medium :high :very-high))

(def TypeClass String)
(def TypeHints {(s/optional-key :cardinality)    Cardinality
                (s/optional-key :selectivity)    Selectivity
                (s/optional-key :class)          TypeClass
                (s/optional-key :collection)     Boolean
                (s/optional-key :case-sensitive) Boolean
                (s/optional-key :globbable)      Boolean
                (s/optional-key :tokenized)      Boolean})
(def Type Keyword)
(def Types {Type TypeHints})
(def Field Keyword)

;WIP
(def FieldInfo {Field (merge {:type Type} TypeHints)})      ;;geting a bit silly?
(def Views {Keyword [Keyword]})
(def Partitions [Keyword])
(def Indexes [Keyword])

(defrecord Model [fields views partitions indexes]
  component/Lifecycle
  (start [component]
    (info "Starting model component")
    ;;todo verfy against db schema
    component)
  (stop [component]
    (info "Stopping model component")
    component))

(defn validate-model [model]
  (s/validate Model model)
  (let [fields (set (keys (:fields model)))
        views (:views model)
        partitions (:partitions model)
        indexes (:indexes model)
        check-membership (fn [entity entity-members entity-set]
                           (let [diff (set/difference (set entity-members) entity-set)]
                             (if (seq diff)
                               (throw (Exception. (str "Unknown member(s) " (vec diff) " in " entity))))))]
    ;fields checked against database in Cassandra component
    (doseq [name (keys views)
            :let [view-set (name views)]]
      (check-membership (str "view " name) view-set (set (concat fields (keys views)))))
    (check-membership "partitions" partitions fields)
    (check-membership "indexes" indexes fields)))

(defn expand-fields
  "Walk fields map, inline type info from types map, assoc-ing :type <type> as we do it
  transforming:
  {:f1 :int...}
       =>
  {:f1 {:type :int :cardinality :low .. }..}"
  [fields-conf type-conf]
  (reduce
    (fn [walked-fields [field type]]
      (assoc walked-fields field (assoc (type type-conf) :type type)))
    {}
    fields-conf))

(defn expand-views
  "walk view definitions, recursive inlining view references as fields
  transforming:
  { :v1 [:a]
    :v2 [:v1 :b]
    :v3 [:v2 :c] ..}
=> {:v1 (:a), :v2 (:a :b), :v3 (:a :b :c), }"
  [views]
  (letfn [(expand [elts]
                  (map
                    (fn [elt]
                      (if (some (set (keys views)) [elt])
                        (expand (views elt)) elt))
                    elts))]
    (zipmap (keys views) (for [elts (vals views)]
                           (distinct (flatten (expand elts)))))))

(defn new-model [model-conf-or-model-path]
  (let [types-conf (edn/read-string (slurp (io/resource default-types-jarpath)))
        model-conf (if (map? model-conf-or-model-path) model-conf-or-model-path
                                                       (edn/read-string (slurp (io/resource model-conf-or-model-path))))]
    (s/validate Types types-conf)
    (info "Types loaded from " default-types-jarpath)
    (debug "Types is " types-conf)
    (if-not (map? model-conf-or-model-path)
      (info "Model being loaded from " model-conf-or-model-path))
    (let [model (map->Model (assoc model-conf
                              :fields (expand-fields (:fields model-conf) types-conf)
                              :views (expand-views (:views model-conf))))]
      (debug "Model is " model-conf)
      (validate-model model)
      model)))

(defn cql_ize [f] (keyword (str/replace (name f) \- \_)))
(defn decql_ize [f] (keyword (str/replace (name f) \_ \-)))

(defn case-sensitive? [this field]
  ;ugh these might be cql-ized, todo switch to string fields
  (-> (:fields this) ((decql_ize field)) :case-sensitive true?))

(defn globbable? [this field]
  ;ugh these might be cql-ized, todo switch to string fields
  (-> (:fields this) ((decql_ize field)) :globbable true?))

(defn tokenized? [this field]
  ;ugh these might be cql-ized, todo switch to string fields
  (-> (:fields this) ((decql_ize field)) :tokenized true?))
