(ns smx.eventstore.search.executor
  (:require [clojure.tools.logging :refer [info error]]
            [com.stuartsierra.component :as component]
            [clojure.string :as str]
            [smx.eventstore.search.plan-node :as nodes]
            [schema.core :as s]
            [clojure.core.async :as async]
            [smx.eventstore.search.log :as slog]
            [smx.eventstore.search.searches :as searches]
            [clojure.tools.logging :as log])
  (:import [smx.eventstore.search.searches Context Searches]
           [clojure.core.async.impl.protocols Channel]
           [smx.eventstore.search.plan_node PlanNode]))

(s/defrecord Executor [searches :- Searches
                       search-ttl-secs :- Long
                       page-size :- Long]
  ;; Implement the Lifecycle protocol
  component/Lifecycle
  (start [this]
    (info "Starting executor")
    this)
  (stop [this]
    (info "Stopping executor")
    this))

(s/defn new-executor [search-ttl-secs :- Long
                      page-size :- Long]
  (map->Executor {:search-ttl-secs search-ttl-secs
                  :page-size       page-size}))

(defn transform-key [[k v]]
  (if (keyword? k)
    [(keyword (str/replace (name k) "_" "-")) v]
    [k v]))

(defn hyphenate-keys
  [result]
  (into {} (map transform-key result)))

(defn hyphenate-all-keys                                    ;ughh
  [results]
  (for [result results]
    (hyphenate-keys result)))

(s/defn stream-results
  "Returns channels [peek out].
    out will return coll of size n and will close if no more results.
    peek returns the first val in each coming collection as a way to signal more results imminent."
  ([search-id :- Long                                       ;for logging with slog
    page-size :- Long
    ttl :- Long
    chs-ch :- Channel]
    (slog/debug "Search ttl secs" ttl)
    (slog/debug "Page size" page-size)

    (let [timeout (async/timeout (* 1000 ttl))
          peek (async/chan 2)                               ;current batch and start of next
          out (async/chan 1)]
      (async/go-loop [ch (async/<! chs-ch)
                      page []]
        (if ch
          (if-let [v (async/<! ch)]
            (if (ex-data v)                                 ;dont need this?
              (do (async/>! out v)
                  (async/close! peek)
                  (async/close! out))
              (let [coll (conj page (hyphenate-keys v))]
                (log/trace coll ch)
                (if (= (count coll) 1)
                  (async/>! peek v))
                (if-not (= (count coll) page-size)
                  (recur ch coll)

                  (let [[_ c] (async/alts! [[out coll] timeout])] ;chan size 1 so we'll block here often
                    (if (= c timeout)                       ;nobody looked at next page we're done
                      (do
                        (slog/info "Search has expired.")   ;dont send partial just err
                        (async/close! peek)
                        (async/close! out))
                      (recur ch []))))))
            (recur (async/<!! chs-ch) page))
          (do
            (async/>! out page)
            (async/close! peek)
            (async/close! out))))
      [peek out])))


;todo mutable config
(def ^:dynamic page-size 20)
(def ^:dynamic search-ttl-secs 3600)

(s/defn execute-plan [this :- Executor
                      plan-tree :- PlanNode
                      context :- Context]
  ;walk down tree, nodes pulling on nodes below.
  (stream-results (:search-id context) page-size search-ttl-secs
                  (nodes/execute plan-tree context)))

