(ns signal.data-platform-telemetry.errors
  (:require [circleci.rollcage.core :as rollcage]
            [clojure.tools.logging :as log]))

(defn get-client* [{:keys [environment
                           rollbar-token]}]
  (when (and (= "prod" environment)
             (nil? rollbar-token))
    (throw (IllegalArgumentException. "A Rollbar token is required for the production environment")))

  (log/infof "Rollbar token %s" rollbar-token)
  (when (not-empty rollbar-token)
    (log/info "Initialising rollbar client")
    (rollcage/client rollbar-token {:environment environment})))

(def get-client (memoize get-client*))

(defn notify
  "Sends an exception to the Rollbar service.

   The data must be JSON serialisable, i.e. it must NOT contain InputStreams/"
  ([ex] (notify ex {}))
  ([ex data] (notify (get-client) ex data))
  ([client ex data]
   (when client
     (rollcage/error client ex data))))

(defn- ->url [req]
  (let [query-string? (:query-string req)]
    (cond-> (:uri req)
      query-string? (str "?" query-string?))))

(defn wrap-exceptions
  "Report exceptions in Ring HTTP handlers to rollbar."
  [handler]
  (fn [req]
    (try
      (handler req)
      (catch Exception ex
        (notify ex {:url (->url req)})
        (throw ex)))))

(defn- default-exception-handler [client]
  (proxy [Thread$UncaughtExceptionHandler] []
    (uncaughtException [thread e]
      (log/fatal e "An uncaught exception occurred, the service will exit" :thread thread)
      (notify client e {})
      (System/exit 1))))

(defn init!
  "Initialises error handling."
  [{:keys [environment
           rollbar-token
           add-uncaught-exception-handler?] :or
    {add-uncaught-exception-handler? true}}]
  (let [client (get-client {:environment environment
                            :rollbar-token rollbar-token})]
    (when add-uncaught-exception-handler?
      (Thread/setDefaultUncaughtExceptionHandler (default-exception-handler client))
      (log/info "Registered uncaught exception handler"))))
