(ns simulflow.command
  (:require
   [hato.client :as http]
   [simulflow.utils.core :refer [without-nils]]
   [simulflow.utils.request :refer [sse-request]]))

;;; Command System Utilities
;;; These utilities support the Data Centric Async Processor Pattern
;;; by processing commands generated by transform functions

(defn make-command
  "Create a command data structure for side effects"
  ([kind data]
   {:command/kind kind
    :command/data data})
  ([kind data id]
   {:command/kind kind
    :command/data data
    :command/id id}))

(defn sse-request-command
  "Create an SSE request command for streaming API calls"
  ([params]
   (make-command :command/sse-request (without-nils params)))
  ([url method body & {:keys [headers timeout-ms buffer-size id]}]
   (make-command :command/sse-request
                 (without-nils {:url url
                                :method method
                                :body body
                                :headers headers
                                :timeout-ms timeout-ms
                                :buffer-size buffer-size})
                 id)))

(defn http-request-command
  "Create an HTTP request command for regular API calls"
  [url method body & {:keys [headers timeout-ms id]}]
  (make-command :command/http-request
                (without-nils {:url url
                               :method method
                               :body body
                               :headers headers
                               :timeout-ms timeout-ms})
                id))

;;; Command Handlers
;;; These functions process commands by executing the side effects

(defmulti handle-command
  "Execute a command by dispatching on its kind"
  (fn [command] (:command/kind command)))

(defmethod handle-command :command/sse-request
  [command]
  (let [{:keys [url method body headers timeout-ms buffer-size]} (:command/data command)]
    (:body (sse-request {:request (without-nils {:url url
                                                 :method method
                                                 :body body
                                                 :headers headers
                                                 :timeout-ms timeout-ms})
                         :params (without-nils {:buffer-size buffer-size
                                                :stream/close? true})}))))

(defmethod handle-command :command/http-request
  [command]
  (let [{:keys [url method body headers timeout-ms]} (:command/data command)]
    (http/request (without-nils {:url url
                                 :method method
                                 :body body
                                 :headers headers
                                 :timeout timeout-ms}))))

(defmethod handle-command :default
  [command]
  (throw (ex-info "Unknown command kind" {:command command})))
