(ns elasticsearch-bolt.storm
(:use [elasticsearch-bolt.fields :only
       [elasticsearch-write-output-fields
        elasticsearch-read-output-fields
        elasticsearch-update-output-fields
        elasticsearch-update-with-script-output-fields]]
      [elasticsearch-bolt.core :only
       [index-bulk
        update-doc-with-script
        update-docs
        get-doc
        get-native-doc]]
      [backtype.storm clojure config log]
      [clojure.string :refer [split]])
(:require [clojurewerkz.elastisch.native :as esn]
          [clojurewerkz.elastisch.rest :as esr]
          [clojure.pprint :refer [pprint]]))

(defn settings [cluster-name]
  {"client.transport.ignore_cluster_name" false
   "cluster.name"                         cluster-name
   "client.transport.sniff"               false})

(defn try-index
  [index doctype docs]
  (log-message ">>> try-index connectedNodes: " (.connectedNodes esn/*client*))
  (try (index-bulk index doctype docs)
       (catch Exception e
         (do (log-warn "Exception during index-bulk: " (.getMessage e))
             nil))))

(defn write-success?
  [m]
  (boolean (or (get-in m [:ok])
               (get-in m [:index :ok])
               (get-in m [:create :ok]))))

(defbolt elasticsearch-write
  elasticsearch-write-output-fields {:prepare true :params [hosts cluster-name]}
  [conf context collector]
  (let [_ (log-message "Connecting to " (vec hosts))
        settings (settings cluster-name)
        _ (log-message "Using settings " settings)
        conn (esn/connect! hosts settings)
        _ (log-message ">>> bolt connectedNodes: " (.connectedNodes esn/*client*))]
    (bolt
      (execute
        [tuple]
        (let [{:keys [meta index doctype docs]} tuple
              {:keys [items] :as result} (try-index index doctype docs)]
          (cond
            (nil? result)
            (fail! collector tuple)

            (:has-failures? result)
            (do
              (log-warn (format "Not every doc was indexed successfully for %s/%s! Check elasticsearch logs for more info. Failing tuple. Result: %s\n"
                                index doctype (with-out-str (clojure.pprint/pprint result))))
              (fail! collector tuple))

            :else ;; success
            (let [output [meta :success]]
              (log-message (format "Succeeded in indexing %s documents."
                                   (count items)))
              (emit-bolt! collector output :anchor tuple)
              (ack! collector tuple))))))))

(defbolt elasticsearch-read
   elasticsearch-read-output-fields {:prepare true :params [hosts cluster-name]}
   [conf context collector]
   (let [_ (log-message "Connecting to " (vec hosts))
         settings (settings cluster-name)
         _ (log-message "Using settings " settings)
         conn (esn/connect! hosts settings)
         _ (log-message ">>> bolt connectedNodes: " (.connectedNodes esn/*client*))]
     (bolt
       (execute
         [tuple]
         (let [{:keys [meta index doctype id]} tuple
               results (get-native-doc index doctype id)]
           (if results
             (let [output [meta results]]
               (emit-bolt! collector output :anchor tuple)
               (ack! collector tuple))
             (do
               (log-warn (str "Document does not exist with id: " id))
               (fail! collector tuple))))))))

(defbolt elasticsearch-update
   elasticsearch-update-output-fields {:prepare true :params [uri]}
   [conf context collector]
   (let [_ (log-message "Connecting to " uri)
         conn (esr/connect! uri)]
     (bolt
       (execute
         [tuple]
         (let [{:keys [meta index doctype docs]} tuple
               results (update-docs index doctype docs)
               successful? (every? true? (map write-success? results))]
           (if successful?
             (let [output [meta :success]]
               (log-message (format "Succeeded in updating %s documents."
                                    (count results)))
               (emit-bolt! collector output :anchor tuple)
               (ack! collector tuple))
             (do
               (log-warn "Failed to update document: "
                         (with-out-str (pprint results)))
               (fail! collector tuple))))))))

(defbolt elasticsearch-update-with-script
   elasticsearch-update-with-script-output-fields {:prepare true :params [uri]}
   [conf context collector]
   (let [_ (log-message "Connecting to " uri)
         conn (esr/connect! uri)]
     (bolt
       (execute
         [tuple]
         (let [{:keys [meta index doctype id script params]} tuple
               result (update-doc-with-script index doctype id script params)]
           (cond
             (true? (:ok result))
             (let [output [meta :success]]
               (log-message (format "Succeeded in updating document."))
               (emit-bolt! collector output :anchor tuple)
               (ack! collector tuple))
             (:exception result)
             (do
               (log-warn "Exception while updating document:\n "
                         (with-out-str (pprint (.getMessage (:exception result)))))
               (fail! collector tuple))
             :else
             (do
               (log-warn "Failed to update document: "
                         (with-out-str (pprint result)))
               (fail! collector tuple))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 'Constrcutors'
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn- get-or-throw [m k]
  (let [result (get m k :not-found)]
    (if (= result :not-found)
      (throw (Exception. (format "Key %s was not found in properties: %s" k m)))
      result)))

(defn parse-es-hosts
  "Parses a serialized string of hosts of the form [<host>:<port>,]*"
  [host-specs]
  (map #(let [parsed-ip (split % (re-pattern ":"))]
         (vector (first parsed-ip) (Integer/parseInt (second parsed-ip))))
       (split host-specs (re-pattern ","))))

(defn get-es-hosts [properties]
  (parse-es-hosts (get-or-throw properties "ELASTICSEARCH_HOSTS")))

(defn mk-es-read-bolt [properties]
  ;; Favor IPs over URL.
  (log-message "properties" properties)
  (let [cluster-name (get properties "ELASTICSEARCH_CLUSTER_NAME")
        hosts (get properties "ELASTICSEARCH_HOSTS" nil)
        uri-or-ips (if hosts
                     (vec (parse-es-hosts hosts))
                     (vec (get-or-throw properties "ELASTICSEARCH_URL")))]
    (log-message "Getting elasticsearch host (s): " uri-or-ips)
    (elasticsearch-read uri-or-ips cluster-name)))

(defn mk-es-write-bolt [properties]
  ;; Favor IPs over URL.
  (let [cluster-name (get properties "ELASTICSEARCH_CLUSTER_NAME")
        hosts (get properties "ELASTICSEARCH_HOSTS" nil)
        uri-or-ips (if hosts
                     (vec (parse-es-hosts hosts))
                     (vec (get-or-throw properties "ELASTICSEARCH_URL")))]
    (log-message "Getting elasticsearch host(s): " uri-or-ips)
    (elasticsearch-write uri-or-ips cluster-name)))

(defn mk-es-update-bolt [properties]
  (let [uri (get-or-throw properties "ELASTICSEARCH_URL")]
    (log-message "getting es uri: " uri)
    (elasticsearch-update uri)))

(defn mk-es-update-with-script-bolt [properties]
  (let [uri (get-or-throw properties "ELASTICSEARCH_URL")]
    (log-message "getting es uri: " uri)
    (elasticsearch-update-with-script uri)))
