(ns elasticsearch-bolt.core
  (:use [elasticsearch-bolt.fields :only [elasticsearch-write-output-fields]])
  (:require [clj-http.client :as client]
            [cheshire.core :as json]
            [clojurewerkz.elastisch.rest :as rest]
            [clojurewerkz.elastisch.query :as q]
            [clojurewerkz.elastisch.native.document :as esdn]
            [clojurewerkz.elastisch.rest.bulk :as bulk]
            [clojurewerkz.elastisch.rest.document :as esd]
            [clojurewerkz.elastisch.rest.index :as esi]
            [clojure.string :refer [join]]
            [clojure.walk :as walk]))

;; TODO: change clojurewerkz.elastisch.rest.document and clojurewerkz.elastisch.rest.index to native

(defn create-index [index]
  (esi/create index))

(defn create-mapping [index mappings]
  (esi/create index :mappings mappings))

(defn update-mapping [^String index ^String doctype mapping]
  (esi/update-mapping index doctype :mapping mapping))

;; not available in native
(defn get-mapping [index]
  (esi/get-mapping index))

(defn delete-mapping [index doctype]
  (esi/delete-mapping index doctype))

(defn delete-index [index]
  (esi/delete index))

(defn get-doc [index doctype id]
  (esd/get index doctype id) )

(defn get-native-doc [index doctype id]
  (esdn/get index doctype id) )

(defn search
  "Example usage: (esd/search conn index doctype :query (q/term :title \"kill\")))"
  [index doctype query]
  (esd/search index doctype :query query))

;; TODO native or rest?
#_(defn search
  "Example usage: (esd/search conn index doctype :query (q/term :title \"kill\")))"
  [index doctype query]
  (esdn/search index doctype :query query))

(defn create-doc [index doctype id doc]
  (esd/put index doctype id doc))

(defn index-doc [index doctype id doc]	
 (esd/put index doctype id doc))

(defn prep-doc
  "Prepares the doc by adding the _index and _type keys to the hashmap, for use by Elastisch"
  [doc index doctype]
  (assoc doc
         :_index index
         :_type doctype))

(defn index-bulk [index doctype docs]
  "Indexes the docs in ElasticSearch"
  (let [prepped-docs (map #(prep-doc % index doctype) docs)]
    (bulk/bulk (bulk/bulk-index prepped-docs)
               :refresh true)))

(defn update-doc-with-script
  [index doctype id script params]
  (try (esd/update-with-script index doctype id script params)
       (catch Exception e {:exception e})))

;; There were issues with elastisch's post method. This is solved in later versions of Elastisch.
;; We only need custom 'post' and 'update-with-partial-doc' functions with client version 1.4
(defn post
  [^String uri body]
  (clojure.walk/keywordize-keys (io! (json/decode (:body (client/post uri {:accept :json :body (json/encode body)}))))))

(defn update-with-partial-doc
  "Updates an existing document in the search index with given partial document"
  ([index mapping-type id partial-doc]
   (post (rest/record-update-url index mapping-type id) {:doc partial-doc, :doc_as_upsert true})))

(defn update-doc [index doctype id doc]
  (update-with-partial-doc index doctype id doc))

(defn update-docs [index doctype docs]
  (for [doc docs]
    (update-doc index doctype (:_id doc) doc)))



