(ns elasticsearch-bolt.storm
(:use [elasticsearch-bolt.fields :only
       [elasticsearch-write-output-fields
        elasticsearch-read-output-fields
        elasticsearch-update-output-fields
        elasticsearch-update-with-script-output-fields]]
      [elasticsearch-bolt.core :only
       [index-bulk
        index-nbulk
        update-doc-with-script
        update-doc-with-script-native
        update-docs
        get-doc
        get-native-doc]]
      [backtype.storm clojure config log]
      [clojure.string :refer [split trim]])
(:require [clojurewerkz.elastisch.native :as esn]
          [clojurewerkz.elastisch.rest :as esr]
          [clojure.pprint :refer [pprint]]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Write Native
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn settings [cluster-name]
  {"client.transport.ignore_cluster_name" false
   "cluster.name"                         cluster-name
   "client.transport.sniff"               false})

(defn try-index-native
  [index doctype docs]
  (log-message ">>> try-index connectedNodes: " (.connectedNodes esn/*client*))
  (try (index-nbulk index doctype docs)
       (catch Exception e
         (do (log-warn "Exception during index-bulk: " (.getMessage e))
             nil))))

(defbolt elasticsearch-write-native
  elasticsearch-write-output-fields {:prepare true :params [nodes cluster-name]}
  [conf context collector]
  (let [_ (log-message "Connecting to " (vec nodes))
        settings (settings cluster-name)
        _ (log-message "Using settings " settings)
        conn (esn/connect! nodes settings)
        _ (log-message ">>> bolt connectedNodes: " (.connectedNodes esn/*client*))]
    (bolt
      (execute
        [tuple]
        (let [{:keys [meta index doctype docs]} tuple
              {:keys [items] :as result} (try-index-native index doctype docs)]
          (cond
            (nil? result)
            (fail! collector tuple)

            (:has-failures? result)
            (do
              (log-warn (format "Not every doc was indexed successfully for %s/%s! Check elasticsearch logs for more info. Failing tuple. Result: %s\n"
                                index doctype (with-out-str (clojure.pprint/pprint result))))
              (fail! collector tuple))

            :else ;; success
            (let [output [meta :success]]
              (log-message (format "Succeeded in indexing %s documents."
                                   (count items)))
              (emit-bolt! collector output :anchor tuple)
              (ack! collector tuple))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Write Rest
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn write-success?
  [m]
  (boolean (or (get-in m [:ok])
               (get-in m [:index :ok])
               (get-in m [:create :ok]))))

(defbolt elasticsearch-write-rest
   elasticsearch-write-output-fields {:prepare true :params [uri]}
   [conf context collector]
   (let [_ (log-message "Connecting to " uri)
         conn (esr/connect! uri)]
     (bolt
       (execute
         [tuple]
         (let [{:keys [meta index doctype docs]} tuple
               {:keys [items] :as result} (index-bulk index doctype docs)
               successful? (every? true? (map write-success? items))]
           (if successful?
             (let [output [meta :success]]
               (log-message (format "Succeeded in indexing %s documents."
                                    (count items)))
               (emit-bolt! collector output :anchor tuple)
               (ack! collector tuple))
             (do
               (log-warn (format "Not every doc was indexed successfully for %s/%s! Check elasticsearch logs for more info. Failing tuple. Result: %s\n"
                                 index doctype (with-out-str (clojure.pprint/pprint result))))
               (fail! collector tuple))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Read Native
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defbolt elasticsearch-read-native
   elasticsearch-read-output-fields {:prepare true :params [nodes cluster-name]}
   [conf context collector]
   (let [_ (log-message "Connecting to " (vec nodes))
         settings (settings cluster-name)
         _ (log-message "Using settings " settings)
         conn (esn/connect! nodes settings)
         _ (log-message ">>> bolt connectedNodes: " (.connectedNodes esn/*client*))]
     (bolt
       (execute
         [tuple]
         (let [{:keys [meta index doctype id]} tuple
               results (get-native-doc index doctype id)]
           (if results
             (let [output [meta results]]
               (emit-bolt! collector output :anchor tuple)
               (ack! collector tuple))
             (do
               (log-warn (str "Document does not exist with id: " id))
               (fail! collector tuple))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Update With Doc Rest
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defbolt elasticsearch-update-rest
   elasticsearch-update-output-fields {:prepare true :params [uri]}
   [conf context collector]
   (let [_ (log-message "Connecting to " uri)
         conn (esr/connect! uri)]
     (bolt
       (execute
         [tuple]
         (let [{:keys [meta index doctype docs]} tuple
               results (update-docs index doctype docs)
               successful? (every? true? (map write-success? results))]
           (if successful?
             (let [output [meta :success]]
               (log-message (format "Succeeded in updating %s documents."
                                    (count results)))
               (emit-bolt! collector output :anchor tuple)
               (ack! collector tuple))
             (do
               (log-warn "Failed to update document: "
                         (with-out-str (pprint results)))
               (fail! collector tuple))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Update With Script Native
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defbolt elasticsearch-update-with-script-native
   elasticsearch-update-with-script-output-fields {:prepare true :params [nodes cluster-name]}
   [conf context collector]
   (let [_ (log-message "Connecting to " (vec nodes))
         settings (settings cluster-name)
         _ (log-message "Using settings " settings)
         conn (esn/connect! nodes settings)
         _ (log-message ">>> bolt connectedNodes: " (.connectedNodes esn/*client*))]
     (bolt
       (execute
         [tuple]
         (let [{:keys [meta index doctype id script params]} tuple
               result (update-doc-with-script-native index doctype id script params)]
           (cond
             (true? (:ok result))
             (let [output [meta :success]]
               (log-message (format "Succeeded in updating document."))
               (emit-bolt! collector output :anchor tuple)
               (ack! collector tuple))
             (:exception result)
             (do
               (log-warn "Exception while updating document:\n "
                         (with-out-str (pprint (.getMessage (:exception result)))))
               (fail! collector tuple))
             :else
             (do
               (log-warn "Failed to update document: "
                         (with-out-str (pprint result)))
               (fail! collector tuple))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Update With Script Rest
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defbolt elasticsearch-update-with-script-rest
   elasticsearch-update-with-script-output-fields {:prepare true :params [uri]}
   [conf context collector]
   (let [_ (log-message "Connecting to " uri)
         conn (esr/connect! uri)]
     (bolt
       (execute
         [tuple]
         (let [{:keys [meta index doctype id script params]} tuple
               result (update-doc-with-script index doctype id script params)]
           (cond
             (true? (:ok result))
             (let [output [meta :success]]
               (log-message (format "Succeeded in updating document."))
               (emit-bolt! collector output :anchor tuple)
               (ack! collector tuple))
             (:exception result)
             (do
               (log-warn "Exception while updating document:\n "
                         (with-out-str (pprint (.getMessage (:exception result)))))
               (fail! collector tuple))
             :else
             (do
               (log-warn "Failed to update document: "
                         (with-out-str (pprint result)))
               (fail! collector tuple))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 'Constrcutors'
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn get-or-throw [m k]
  (let [result (get m k :not-found)]
    (if (= result :not-found)
      (throw (Exception. (format "Key %s was not found in conf: %s" k m)))
      result)))

(defn get-rest-settings [conf]
  (get-or-throw conf "ELASTICSEARCH_URL"))

(defn parse-es-nodes
  "Parses a serialized string of hosts of the form [<host>:<port>,]*"
  [host-specs]
  (map #(let [[host port] (map trim (split % #":"))]
         [host (Integer/parseInt port)])
       (split host-specs #",")))

(defn get-native-settings [conf]
  (let [nodes (parse-es-nodes (get-or-throw conf "ELASTICSEARCH_HOSTS"))
        cluster-name (get-or-throw conf "ELASTICSEARCH_CLUSTER_NAME")]
    [nodes cluster-name]))

(defn mk-es-write-native [conf]
  (let [[nodes cluster-name :as settings] (get-native-settings conf)]
    (log-message "mk-es-write-bolt: " (with-out-str (pprint settings)))
    (elasticsearch-write-native nodes cluster-name)))

(defn mk-es-write-rest [conf]
  (let [uri (get-rest-settings conf)]
    (log-message "mk-es-write-bolt-rest: " uri)
    (elasticsearch-write-rest uri)))

(defn mk-es-read-native [conf]
  (let [[nodes cluster-name :as settings] (get-native-settings conf)]
    (log-message "mk-es-read-bolt: " (with-out-str (pprint settings)))
    (elasticsearch-read-native nodes cluster-name)))

(defn mk-es-update-rest [conf]
  (let [uri (get-rest-settings conf)]
    (log-message "mk-es-update-bolt: " uri)
    (elasticsearch-update-rest uri)))

(defn mk-es-update-with-script-native [conf]
  (let [[nodes cluster-name :as settings] (get-native-settings conf)]
    (log-message "mk-es-update-with-script-native: " (with-out-str (pprint settings)))
    (elasticsearch-update-with-script-native nodes cluster-name)))

(defn mk-es-update-with-script-rest [conf]
  (let [uri (get-rest-settings conf)]
    (log-message "mk-es-update-with-script-native: " uri)
    (elasticsearch-update-with-script-rest uri)))
