(ns elasticsearch-bolt.trim
  (:require [clojure.set :as set])
  )

(defn dissoc-vector
  "dissoc[iate]. if it encounters a vector, it will dissoc on each member of that vector."
  ([m] m)
  ([m k]
   (if (map? m)
     ; standard defn: access compiled code
     (. clojure.lang.RT (dissoc m k))
     (map #(dissoc % k) m)))
  ([map key & ks]
   (let [ret (dissoc map key)]
     (if ks
       (recur ret (first ks) (next ks))
       ret))))

(defn dissoc-in
  "Dissociates an entry from a nested associative structure returning a new
  nested structure. keys is a sequence of keys. Any empty maps that result
  will not be present in the new structure."
  [m [k & ks :as keys]]
  (if ks
    (if-let [nextmap (get m k)]
      (if (map? nextmap)
        (let [newmap (dissoc-in nextmap ks)]
          (if (seq newmap)
            (assoc m k newmap)
            (dissoc-vector m k)))
        (let [newmaps (for [x nextmap]
                        (dissoc-in x ks))]
          (assoc m k newmaps)
          )
        )
      m)
    (dissoc-vector m k)))


(defn dissoc-in-recursive
  "takes a vector of vectors and calls out to dissoc-in multiple times"
  [m vec-keys]
  (if (empty? vec-keys)
    m
    (let [ret (dissoc-in m (first vec-keys))]
        (recur ret (rest vec-keys)))))

(defn keys-in [m]
  (if (map? m)
    (vec
      (mapcat (fn [[k v]]
                (let [sub (keys-in v)
                      nested (map #(into [k] %) (filter (comp not empty?) sub))]
                  (if (seq nested)
                    nested
                    [[k]])))
              m))
    (if (or (vector? m)
            (seq? m))
      (keys-in (first m))
      [])
    ))

(defn make-walker [f]
  (fn my-fn [m]
       (let [pairs (for [[k v] m]
                     (f my-fn k v))
             keypairs (flatten pairs)
             final (if (> (count keypairs) 0)
                     (apply assoc m keypairs)
                     m)]
         final)))

(defn mapping-to-template
  "Takes in a mapping and returns a map that can be used to figure out which"
  [mapping]
  (let [peeled (:properties mapping)
        traverse (make-walker (fn [recurse k v]
                                (if (:properties v)
                                  [k (recurse (:properties v))]
                                  [k 1])))]
    (traverse peeled)))

 (defn trim-activity-for-elasticsearch [activity template]
   (let [activity-keys (keys-in activity)
         template-keys (keys-in template)

         keys-to-delete (set/difference
                          (set activity-keys)
                          (set template-keys))]
     (dissoc-in-recursive activity keys-to-delete)))

(defn strip-map-to-relevant-keys
  "Takes an Elasticsearch mapping and a document that's going into the mapping.
  Returns a doc that has been stripped to only the keys which will be stored in
  Elasticsearch."
  [doc mapping]
  (trim-activity-for-elasticsearch doc
                                   (mapping-to-template mapping)))

