(ns elasticsearch-bolt.storm
  (:use [elasticsearch-bolt.fields :only
         [elasticsearch-write-output-fields
          elasticsearch-update-output-fields
          elasticsearch-update-with-script-output-fields]]
        [elasticsearch-bolt.core :only
         [index-bulk
          update-doc-with-script
          update-docs]]
        [backtype.storm clojure config log])
  (:require [clojurewerkz.elastisch.rest :as esr]))

(defn wrtie-success?
  [m]
  (boolean (or (get-in m [:ok])
               (get-in m [:index :ok])
               (get-in m [:create :ok]))))

(defbolt elasticsearch-write
  elasticsearch-write-output-fields {:prepare true :params [uri]}
  [conf context collector]
  (let [_ (log-message "Connecting to " uri)
        conn (esr/connect! uri)]
    (bolt
      (execute
        [tuple]
        (let [{:keys [meta index doctype docs]} tuple
              {:keys [items] :as result} (index-bulk index doctype docs)
              successful? (every? true? (map wrtie-success? items))]
          (if successful?
            (let [output [meta :success]]
              (log-message (format "Succeeded in indexing %s documents."
                                   (count items)))
              (emit-bolt! collector output :anchor tuple)
              (ack! collector tuple))
            (do
              (log-warn (format "Not every doc was indexed successfully for %s/%s! Check elasticsearch logs for more info. Failing tuple. Result: %s\n"
                                index doctype (with-out-str (clojure.pprint/pprint result))))
              (fail! collector tuple))))))))

(defbolt elasticsearch-update
   elasticsearch-update-output-fields {:prepare true :params [uri]}
   [conf context collector]
   (let [_ (log-message "Connecting to " uri)
         conn (esr/connect! uri)]
     (bolt
       (execute
         [tuple]
         (let [{:keys [meta index doctype docs]} tuple
               results (update-docs index doctype docs)
               successful? (every? true? (map wrtie-success? results))
               ]
           (if successful?
             (let [output [meta :success]]
               (log-message (format "Succeeded in updating %s documents."
                                    (count results)))
               (emit-bolt! collector output :anchor tuple)
               (ack! collector tuple))
             (do
               (log-warn "Failed to update document: "
                         (with-out-str (clojure.pprint/pprint results)))
               (fail! collector tuple))))))))

(defbolt elasticsearch-update-with-script
   elasticsearch-update-with-script-output-fields {:prepare true :params [uri]}
   [conf context collector]
   (let [_ (log-message "Connecting to " uri)
         conn (esr/connect! uri)]
     (bolt
       (execute
         [tuple]
         (let [{:keys [meta index doctype id script params]} tuple
               result (update-doc-with-script index doctype id script params)]
           (if (true? (:ok result))
             (let [output [meta :success]]
               (log-message (format "Succeeded in updating document."))
               (emit-bolt! collector output :anchor tuple)
               (ack! collector tuple))
             (do
               (log-warn "Failed to update document: "
                         (with-out-str (clojure.pprint/pprint result)))
               (fail! collector tuple))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 'Constrcutors'
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn- get-or-throw [m k]
  (let [result (get m k :not-found)]
    (if (= result :not-found)
      (throw (Exception. (format "Key %s was not found in properties: %s" k m)))
      result)))

;;; DEPRECATED: use mk-es-write-bolt
(defn make-elasticsearch-bolt [properties]
  (let [uri (get-or-throw properties "ELASTICSEARCH_URL")]
    (log-message "getting es uri: " uri)
    (elasticsearch-write uri)))

(defn mk-es-write-bolt [properties]
  (let [uri (get-or-throw properties "ELASTICSEARCH_URL")]
    (log-message "getting es uri: " uri)
    (elasticsearch-write uri)))

(defn mk-es-update-with-script-bolt [properties]
  (let [uri (get-or-throw properties "ELASTICSEARCH_URL")]
    (log-message "getting es uri: " uri)
    (elasticsearch-update-with-script uri)))
