(ns elasticsearch-bolt.core
  (:use [elasticsearch-bolt.fields :only [elasticsearch-output-fields]])
  (:require
    [clojurewerkz.elastisch.query :as q]
    [clojurewerkz.elastisch.rest.bulk :as bulk]
    [clojurewerkz.elastisch.rest.document :as esd]
    [clojurewerkz.elastisch.rest.index :as esi]
    [clojure.walk :as walk]
    ))

(defn create-index [index]
  (esi/create index))

(defn create-mapping [index mappings]
  (esi/create index :mappings mappings))

(defn update-mapping [^String index ^String doctype mapping]
  (esi/update-mapping index doctype :mapping mapping))

(defn get-mapping [index]
  (esi/get-mapping index))

(defn delete-mapping [index doctype]
  (esi/delete-mapping index doctype))

(defn delete-index [index]
  (esi/delete index))

(defn get-doc [index doctype id]
  (esd/get index doctype id) )

(defn search
  "Example usage: (esd/search conn index doctype :query (q/term :title \"kill\")))"
  [index doctype query]
  (esd/search index doctype :query query))

(defn prep-doc
  "Prepares the doc by adding the _index and _type keys to the hashmap, for use by Elastisch"
  [doc index doctype]
  (assoc doc
         :_index index
         :_type doctype))

(defn index-bulk [index doctype docs]
  "Indexes the docs in ElasticSearch"
  (let [prepped-docs (map #(prep-doc % index doctype) docs)]
    (bulk/bulk (bulk/bulk-index prepped-docs)
               :refresh true)))

(defn index-doc [index doctype id doc]
  (esd/put index doctype id doc))

; (defn index-docs [index doctype docs]
;   (for [doc docs]
;     (index-doc index doctype (:_id doc) doc)))
