(ns csv-export-bolt.storage.s3
  (:require [backtype.storm.log :as storm] 
            [amazonica.aws.s3 :as s3]
            [cheshire.core :as json]
            [clojure.java.io :as io]
            [clojure.string :as st]))


(defn get-or-throw [conf key]
  (or (get conf key)
      (throw (Exception. (str "Missing required config field: " key)))) )

(defn mk-credentials
  [conf]
  {:access-key (get-or-throw conf "AWS_ACCESS_KEY_ID")
   :secret-key (get-or-throw conf "AWS_SECRET_ACCESS_KEY")
   :endpoint   (get-or-throw conf "AWS_S3_REGION")})

(defn put-object
  [creds bucket location file]
  (s3/put-object creds {:bucket-name bucket :key location :file file})
  (str "s3://" bucket "/" location))

(defn safe-put
  "Attempt to PUT the file to s3 returns full s3 path when successful or 
   nil if unsuccessful. Retries on failure up to max-retries times."
  [creds bucket location file
   & {:keys [retry-count max-retries wait-time]
      :or {retry-count 0, max-retries 10 wait-time 1000}}]
  ;; Store the file and return the location
  (try
    (put-object creds bucket location file) 
    (catch Exception e
      (do (Thread/sleep wait-time)
          (storm/log-error e " Failed to store in s3. "
                           "Retry count: " retry-count)
          (if (< retry-count max-retries)
            (safe-put creds bucket location file
                      :retry-count (inc retry-count)
                      :max-retries max-retries
                      :wait-time wait-time)
            (storm/log-warn "safe-put failed to store to s3 after "
                            max-retries " attempts for " bucket location))))))

(defn store-file
  [conf location file-path]
  (let [ ;; For backwards compatibility look for the old key as fallback
        bucket (or (get conf "ARCHIVE_WRITE_S3_BUCKET")
                   (get conf "S3_BUCKET")
                   (throw (Exception. "Missing config field ARCHIVE_WRITE_S3_BUCKET")))
        creds (mk-credentials conf)
        file (io/file file-path)]
    (safe-put creds bucket location file)))
