(ns readme-utilities
  "trlisp-specific functions for adjusting ReadMoi output."
  (:require [tree-calculus.utilities :refer [tree->nat]]))


(defn- replace-something
  "Given string `s`, replaces all occurances of regex `re`, with `replacement`."
  {:UUIDv4 #uuid "21de6057-6189-47a7-a297-fc247c319cd3"}
  [s re replacement]
  (clojure.string/replace s re replacement))


(defn replace-leaf
  "Given string `s`, replace all occurances of `[]` with `Δ`."
  {:UUIDv4 #uuid "eb576c67-ff2d-4926-8ccd-d5e0616b2cfc"}
  [s]
  (replace-something s #"\[\]" "Δ"))


(defn replace-stem
  "Given string `s`, replace all occurances of `[[]]` with `K`."
  {:UUIDv4 #uuid "9b5d700e-3022-4594-905f-ae8ae8a69292"}
  [s]
  (replace-something s #"\[\[\]\]" "K"))


(defn replace-fork
  "Given string `s`, replace all occurances of `[[] []]` with `(K Δ)`."
  {:UUIDv4 #uuid "0d1a7ad2-b05d-454b-8fc1-3e6cbf8d21d7"}
  [s]
  (replace-something s #"\[\[\] \[\]\]" "(K Δ)"))


(defn replace-trees
  "Given a `[:code ...]` block `b` generated by `print-form-then-eval`, replace
  occurances of simple forks, stems, and leafs."
  {:UUIDv4 #uuid "f711416d-07e5-4fbe-8204-fbf22f187a3e"}
  [b]
  (->> b
       second
       replace-fork
       replace-stem
       replace-leaf
       (into [:code])))


(defn replace-true
  "Given string `s`, replace all occurances of `[[]]` with `True`."
  {:UUIDv4 #uuid "664b94b3-f987-4f32-b05c-65007dad0336"}
  [s]
  (replace-something s #"\[\[\]\]" "True"))


(defn replace-false
  "Given string `s`, replace all occurances of `[[] [[[]] [[]]]]"
  [s]
  (replace-something s #"\[\[\] \[\[\[\]\] \[\[\]\]\]\]" "False"))


(defn replace-booleans
  "Given a `[:code ...]` block `b` generated by `print-form-then-eval`, replace
  occurances of True and False."
  {:UUIDv4 #uuid "987c1337-6f1a-4ae2-b779-d681b41ba9f0"}
  [b]
  (->> b
       second
       replace-false
       replace-true
       (into [:code])))


(defn make-leaf
  {:UUIDv4 #uuid "e0abd786-f81f-4251-8b19-4aebdc475e0b"}
  []
  "Δ")


(defn make-stem
  "Given child tree `t`, return a printed version of a stem. If optional
  argument `arg` is given, appends an application symbol and `arg`.

  `.toString` is called on `t` and `arg`."
  {:UUIDv4 #uuid "c32d9362-1959-4542-ab37-85fbcc8ec535"}
  ([t] (make-stem t nil))
  ([t arg]
   (let [child (.toString t)
         left-padding (-> child
                          count
                          (/ 2)
                          Math/floor
                          (repeat " ")
                          clojure.string/join)]
     (str left-padding "Δ" "\n"
          left-padding "|" (when arg (str "  ␣  " arg)) "\n"
          child))))


(defn make-fork
  "Given child trees `left` and `right`, return a printed version of a fork. If
  optional argument `arg` is given, appends an application symbol and `arg`.

  `.toString` is called on `left`, `right`, and `arg`."
  {:UUIDv4 #uuid "6fdb3140-207a-481a-8c8d-2796e2b7961c"}
  ([left right] (make-fork left right nil))
  ([left right arg]
   (let [L (.toString left)
         R (.toString right)
         left-padding (-> L
                          count
                          dec
                          (repeat " ")
                          clojure.string/join)]
     (str left-padding "  Δ" "\n"
          left-padding " /" " " "\\" (when arg (str "   ␣  " arg)) "\n"
          L "   " R))))


(defn make-fork-stem
  "Given child trees `left` and `right`, return a printed version of a fork
  with a left child stem. If optional argument `arg` is given, appends an
  application symbol and `arg`.

  Unlike `make-stem` and `make-fork`, does not include variable padding."
  {:UUIDv4 #uuid "88f325f1-14c6-477b-b2e1-e119261d47e9"}
  ([left right] (make-fork-stem left right nil))
  ([left right arg]
   (str "  Δ" "\n"
        " / \\" "\n"
        "Δ   " right (when arg (str "  ␣  " arg)) "\n"
        "|" "\n"
        left)))


(defn make-fork-fork
  "Given child trees `left`, `middle`, and `right`, return a printed version of
  a fork with a left child fork. If optional argument `arg` is given, appends an
  application symbol and `arg`.

  Unlike `make-stem` and `make-fork`, does not include variable padding."
  {:UUIDv4 #uuid "02fffedc-e713-4a85-a0af-89859262f5b8"}
  ([left middle right] (make-fork-fork left middle right nil))
  ([left middle right arg]
   (str "    Δ" "\n"
        "   / \\" "\n"
        "  Δ   " right (when arg (str "  ␣  " arg)) "\n"
        " / \\" "\n"
        left "   " middle)))


(defn make-nat-number
  "Given natural number `n`, returns a tree representation."
  {:UUIDv4 #uuid "2e1052e8-d6f9-4f78-9348-6d837b8fac72"}
  [n]
  (let [padding #(clojure.string/join (repeat % "  "))
        components (reduce #(concat %1 [[(padding %2) " / \\" "\n"]
                                        [(padding %2) "Δ   Δ"  "\n"]])
                           [["  Δ" "\n"]]
                           (range n))]
    (-> components
        flatten
        clojure.string/join)))


(defn replace-nat-number
  "Given a `[:code ...]` block `b` generated by `print-form-then-eval`, replace
  occurances of natural number trees with the integer itself."
  {:UUIDv4 #uuid "a49692ac-6fe8-4c58-8c0f-d986b51bd917"}
  [b]
  (let [block-contents (second b)
        separator " ;; => "
        blocks (clojure.string/split block-contents (re-pattern separator))]
    [:code (str (first blocks)
                separator
                (-> (second blocks)
                    read-string
                    tree->nat
                    str))]))

