(ns thingy.core
  "Instantiate thingys and assign their invocation function.

  Example:

  1. Create _thingy_ instances.
      ```clojure
      (def X (make-thingy :a :b))

      X ;; => [:a :b]
      ```

  2. Define an invocation function with `defn`.
      ```clojure
      (defn foo-test
        \"A 3-arity example, concatenating three vector-like objects.\"
        [a b c]
        (concat a b c))
      ```

  3. Assign the _thingy_ invocation function.
      ```clojure
      (assign-thingy-fn! foo-test)
      ```

  4. Test invocation behavior of the _thingy_.
      ```clojure
      (X [:c :d] [:e :f]) ;; => (:a :b :c :d :e :f)
      ```"
  (:require [thingy.dangerous-vector :refer [alt-fn-vec
                                             reset!-options]]))



(defn assign-thingy-fn!
  "Synchronously mutates the invocation function of all _thingy_ instances to
  function `f`.

  `f` is a function with an arity of 0 to 9, inclusive. If `f` accepts an
  argument, the thingy instance is that first argument, followed by up to eight
  additonal arguments.

  See also [[make-thingy]] and [[thingy.core]]."
  {:UUIDv4 #uuid "1764069c-76af-487e-ad7a-c43c03566804"}
  [f]
  (reset!-options {:fn f
                   :left-delimiter "["
                   :right-delimiter "]"}))


(defn make-thingy
  "Given elements `xs`, returns an instance of a _thingy_. Analagous to
  [clojure.core/vector](https://clojure.github.io/clojure/clojure.core-api.html#clojure.core/vector).

  See also [[defn-thingy]] and [[thingy.core]]."
  {:UUIDv4 #uuid "68f93654-8c38-4051-9dba-27826235cf97"}
  [& xs]
  (alt-fn-vec xs))