(ns case-study
  "Generate an hiccup/html webpage describing a case study concerning specifying
  and validating the Speculoos project changelog.edn"
  {:no-doc true}
  (:require
   [fn-in.core :refer [get-in*]]
   [hiccup2.core :as h2]
   [hiccup.page :as page]
   [hiccup.element :as element]
   [hiccup.form :as form]
   [hiccup.util :as util]
   [speculoos-hiccup :refer :all]
   [speculoos.core :refer [#_all-paths
                           #_expand-and-clamp-1
                           only-invalid
                           valid-collections?
                           valid-scalars?
                           valid?
                           validate
                           validate-collections
                           validate-scalars]]
   [speculoos.utility :refer [#_*such-that-max-tries*
                              basic-collection-spec-from-data
                              #_clamp-in*
                              collections-without-predicates
                              data-from-spec
                              defpred
                              #_exercise
                              #_inspect-fn
                              in?
                              predicates-without-collections
                              predicates-without-scalars
                              scalars-without-predicates
                              sore-thumb
                              spec-from-data
                              thoroughly-valid?
                              #_unfindable-generators
                              validate-predicate->generator
                              thoroughly-valid-scalars?]]))


(def changelog-data (load-file "resources/case_study/edited_changelog.edn"))


(def page-body
  [:body
   [:article
    [:h1 "Case study: Specifying and validating the Speculoos library changelog"]

    [:section#intro
     [:p "So what's it like to use " [:a {:href "https://github.com/blosavio/speculoos"} "Speculoos"] " on a task that's not merely a demo? Let's specify and validate the Speculoos library " [:code "changelog.edn"] ". To begin, a few words about the changelog itself."]

     [:p "Speculoos is an experimental library. Among the ideas I wanted to explore is a changelog published in Clojure " [:strong "e"] "xtensible " [:strong "d"] "ata " [:strong "n"] "otation (" [:a {:href "https://github.com/edn-format/edn"} "edn"] "). The goal is to have a single, canonical, human- and machine-readable document that describes the project's changes from one version to the next. That way, it would be straightforward to automatically generate a nicely-formatted changelog webpage and query the changelog data so that people can make informed descisions about changing versions."]

     [:p "Here's the info that I think would be useful for a changelog entry."]

     [:ul
      [:li "Version number"]
      [:li "Date"]
      [:li "Person responsible, with contact info"]
      [:li "Status of the project (i.e., stable, deprecated, etc.)"]
      [:li "Urgency (i.e., high for a security fix, etc.)"]
      [:li "Flag if changes are breaking from previous version"]
      [:li "Free-form comments"]]

     [:p "We can quickly assemble an example."]

     [:pre [:code "{:version 99\n :date {:year 2025\n        :month \"November\"\n        :day 12}\n :responsible {:name \"Kermit Frog\"\n               :email \"its.not.easy@being.gre.en\"}\n :project-status :stable\n :urgency :low\n :breaking? false\n :comment \"Improved arithmetic capabilities.\"\n :changes [«see upcoming discussion»]}"]]

     [:p "Furthermore, for each of those changelog entries, I think it would be nice to tell people more details about the individual changes so they can make technically supported decisions about changing versions. A single, published version could consist of multiple changes, associated to a key " [:code ":changes"] ", with each change detailed with this info."]

     [:ul
      [:li "A free-form, textual description of the change."]
      [:li "A reference, (e.g., GitHub issue number, JIRA ticket number, etc.)"]
      [:li "Kind of change (i.e., bug fix, renamed function, removed function, improved performance, etc.)"]
      [:li "Flag indicating if this particular change is breaking."]
      [:li "Added, renamed, moved, altered, or deleted functions."]]

     [:p "Here's an example of one change included in a published version."]

     [:pre [:code "{:description \"Addition function `+` now handles floating point decimal number types.\"\n :reference {:source \"Issue #78\"\n             :url \"https://example.com/issue/87\"}\n :change-type :relaxed-input-requirements\n :breaking? false\n :altered-functions ['+]\n :date {:year 2025\n        :month \"November\"\n        :day 8}\n :responsible {:name \"Fozzie Bear\"\n               :email \"fozzie@wocka-industries.com\"}}"]]

     [:p "The date and person responsible for an individual change need not be the same as the date and person responsible for the version that contains it. So while Kermit was responsible for publishing the overall verison on 2025 November 12, Fozzie was responsible for creating the individual change to the plus function on 2025 November 08."]

     [:p "With the expected shape of our changelog data established, we can now compose the specifications that will allow us to validate the data. We must keep in mind Speculoos ' "[:a {:href "https://github.com/blosavio/speculoos/tree/main?tab=readme-ov-file#-three-mottos"} "Three Mottos" ] ". Motto #1 reminds us to keep scalar a collection specifications separate. The validation functions themselves enforce this principle, but adhering to Motto #1 helps minimize confusion."]

     [:p " Motto #2 reminds us to shape the specification to that it mimics the data. This motto reveals a convenient tactic: copy-paste the data, delete the scalars, and insert predicates."]

     [:p "Motto #3 reminds us to ignore un-paired predicates and un-paired datums. In practice, the consequence of this principle is that we may provide more data than we specify, and the un-specified data merely flows through, un-validated. In the other direction, we may specify more elements than actually exist in a particular piece of data. That's okay, too. Those un-paired predicates will be ignored."]
     
     [:p "Our overall strategy is this: Build up specifications from small pieces, testing those small pieces along the way. Then, we we're confident in the small pieces, we can assemble them at the end. We'll start with specifying and validating the scalars. Once we've done that, we'll put them aside. Then, we'll specify and validate thecollections, testing them until we're confident we've got the correct specifications. At the end, we'll bring together bot scalar validation and collection validation into a combo validation."]

     [:p "I hope the structure of this case study document reinforces the principles we just laid out."]

     [:a {:href "#scalars"} "Specifying & validating scalars"] [:br]
     [:a {:href "#collections"} "Specifying & validating collections"] [:br]
     [:a {:href "#combo"} "Combo validations"]

     [:p "Scalars and collections are separate concepts, so we handle them in different steps. At the end, merely for convenience, we can use a combo validation that separately validates the scalars and the collections with a single invocation."]

     [:p "Let's set up our environment with the tools we'll need."]

     [:pre
      (print-form-then-eval "(require '[speculoos.core :refer [valid-scalars? valid-collections? valid?]])")
      [:br]
      [:br]
      (print-form-then-eval "(require '[fn-in.core :refer [get-in*]])")
      [:br]
      [:br]
      (print-form-then-eval "(set! *print-length* 99)")]]

    
    [:section#scalars
     [:h3 "Specifying & validating scalars"]

     [:p "We'll start simple. Let's compose a " [:em "date"] " specification. Informally, a date is a year, a month, and a day. Let's stipulate that a valid year is " [:em "An integer greater-than-or-equal-to two-thousand"] ". Here's a predicate for that concept."]
     
     [:pre (print-form-then-eval "(defn year-predicate [n] (and (int? n) (<= 2000 n)))")]

     [:p "Speculoos predicates are merely Clojure functions. Let's try it."]

     [:pre
      (print-form-then-eval "(year-predicate 2025)")
      [:br]
      [:br]
      (print-form-then-eval "(year-predicate \"2077\")")]

     [:p "That looks good. Integer " [:code "2025"] " is greater than two-thousand, while string " [:code "\"2077\""] " is not an integer."]

     [:p "Checking day of the month is similar."]

     [:pre (print-form-then-eval "(defn day-predicate [n] (and (int? n) (<= 1 n 31)))")]

     [:p [:code "day-predicate"] " is satisfied only by an integer between one and thirty-one, inclusive."]

     [:p "Speculoos can validate a scalar by testing if it's a member of a set. A valid month may only be one of twelve elements. Let's enumerate the months of the year, months represented as strings."]

     [:pre (print-form-then-eval "(def month-predicate #{\"January\"
                                                          \"February\"
                                                          \"March\"
                                                          \"April\"
                                                          \"May\"
                                                          \"June\"
                                                          \"July\"
                                                          \"August\"
                                                          \"September\"
                                                          \"October\"
                                                          \"November\"
                                                          \"December\"})")]

     [:p "Let's see how that works."]

     [:pre (print-form-then-eval "(month-predicate \"August\")")]

     [:p [:code "month-predicate"] " is satisfied (i.e., returns a truthy value) because string " [:code "\"August\""] " is a member of the set."]

     [:pre (print-form-then-eval "(month-predicate :November)")]

     [:p "Keyword " [:code ":November"] " does not satisfy " [:code "month-predicate"] " because it is not a member of the set. " [:code "month-predicate"] " returns a falsey value, " [:code "nil"] "."]

     [:p "We've now got predicates to check a year, a month, and day. The notion of " [:em "date"] " includes a year, month, and a day travelling around together. We can collect them into one group using a Clojure collection. A hash-map works well in this scenario."]

     [:pre [:code "{:year 2020\n :month \"January\"\n :day 1}"]]

     [:p "Speculoos specifications are plain old regular Clojure data collections. " [:a {:href ""} "Motto #2"] " reminds us to shape the specification to mimic the data. To create a scalar specification, we could copy-paste the data, and delete the scalars…"]

     [:pre [:code "{:year ____\n :month ___ \n :day __}"]]

     [:p "…and insert our predicates."]

     [:pre (print-form-then-eval "(def date-spec {:year year-predicate :month month-predicate :day day-predicate})")]

     [:p "Let's check our progress against some valid data. We're validating scalasrs (Motto #1), so we'll use a function with a " [:code "-scalars"] " suffix. The data is the first argument on the upper row, the specification is the second argument on the lower row."]

     [:pre (print-form-then-eval "(valid-scalars? {:year 2024 :month \"January\" :day 1} {:year year-predicate :month month-predicate :day day-predicate})" 85 75)]

     [:p "Each of the three scalars satisfies their respective predicates (Motto #3), so " [:code "valid-scalars?"] " returns " [:code "true"] "."]

     [:p "Now let's feed in some invalid data."]

     [:pre (print-form-then-eval "(valid-scalars? {:year 2024 :month \"Wednesday\" :day 1} {:year year-predicate :month month-predicate :day day-predicate})" 85 85)]

     [:p "While " [:code "\"Wednesday\""] " is indeed a string, it is not a member of the " [:code "month-predicate"] " set, so " [:code "valid-scalars?"] " returns " [:code "false"] "."]

     [:div.no-display
      [:p "Perhaps we could have used an " [:code "instant"] " literal like this."]

      [:pre (print-form-then-eval "(java.util.Date.)")]

      [:p "But I wanted to demonstrate how Speculoos can specify and validate hand-made date data."]]

     [:p "Now that we can validate the date portion of the changelog, we'll need to specify and validate the information about the person responsible for that publication. The changelog information about a person is their name, a free-form string, and an email address, also a string. In addition to being a string, a valid email address:"]

     [:ul
      [:li "Starts with any number of alphanumeric characters or periods,"]
      [:li "Followed by exactly one " [:code "@"] " character,"]
      [:li "Followed by any number of alphanumeric characters or periods."]]

     [:p "Regualar expressions are powerful tools for testing that kind of string properties, and Speculoos scalar validtion supports them. A regular expression appearing in a scalar specification is considered a predicate. Let's make the following a specification about a changelog person."]

     [:pre (print-form-then-eval "(def person-spec {:name string? :email #\"^[\\w\\.]+@[\\w\\.]+\"})")]

     [:p "Let's give that specificaiton a whirl. First, we validate some valid person data (data in upper row, specification in lower row)."]

     [:pre (print-form-then-eval "(valid-scalars? {:name \"Abraham Lincoln\" :email \"four.score.seven.years@gettysburg.org\"} {:name string? :email #\"^[\\w\\.]+@[\\w\\.]+\"})" 95 95)]

     [:p "Both name and email scalars satisfied their paired predicates. Now, let's see what happens when we validate some data that is invalid."]

     [:pre (print-form-then-eval "(valid-scalars? {:name \"George Washington\" :email \"crossing_at_patomic\"} {:name string? :email #\"^[\\w\\.]+@[\\w\\.]+\"})")]

     [:p "Oops. That email address does not satisfy the regular expression because it does not contain an " [:code "@"] " character, so the data is invalid."]

     [:p "Perhaps the most pivotal single datum in a changelog entry is the version number. For our discussion, let's stipulate that a version is an integer greater-than-or-equal-to zero. Here's a predicate for that."]

     [:pre (print-form-then-eval "(defn version-predicate [i] (and (int? i) (<= 0 i)))")]

     [:p "And a pair of quick demos."]

     [:pre
      (print-form-then-eval "(version-predicate 99)")
      [:br]
      [:br]
      (print-form-then-eval "(version-predicate -1)")]

     [:p "At this point, let's assemble what we have. Speculoos specifications are merely Clojure collections that mimic the shape of the data. So let's collect those predicates into a map."]

     [:pre [:code "{:version version-predicate\n :date date-spec\n :person person-spec}"]]

     [:p "Notice, " [:code "date-spec"] " and " [:code "person-spec"] " are each themselves specifications. We compose a Speculoos specificaiton using standard Clojure composition."]

     [:p "The partial changelog entry might look something like this."]

     [:pre [:code
            "{:version 99\n :date {:year 2025\n        :month \"August\"\n        :day 1}\n :person {:name \"Abraham Lincoln\"\n          :email \"four.score.seven.years@gettysburg.org\"}}"]]

     [:p "Let's check our work so far. First, we'll validate some data we know is valid."]

     [:pre (print-form-then-eval "(valid-scalars? {:version 99 :date {:year 2025 :month \"August\" :day 1} :person {:name \"Abraham Lincoln\" :email \"four.score.seven.years@gettysburg.org\"}} {:version version-predicate :date date-spec :person person-spec})")]

     [:p "Dandy."]

     [:p "Second, we'll feed in some data we suspect is invalid."]

     [:pre (print-form-then-eval "(valid-scalars? {:version 1234 :date {:year 2055 :month \"Octoberfest\" :day 1} :person {:name \"Paul Bunyan\" :email \"babe@blue.ox\"}} {:version version-predicate :date date-spec :person person-spec})")]

     [:p "Hmm. " [:em "Something"] " doesn't satisfy their predicate, but my eyesight isn't great and I can't immediately spot the problem. Let's use a more verbose function, " [:code "validate-scalars"] ", which returns detailed results."]

     [:pre (print-form-then-eval "(validate-scalars {:version 1234 :date {:year 2055 :month \"Octoberfest\" :day 1} :person {:name \"Paul Bunyan\" :email \"babe@blue.ox\"}} {:version version-predicate :date date-spec :person person-spec})")]


     [:p "Ugh, too verbose. Let's pull in a utility that filters the validation results so only the invalid results are displayed."]

     [:pre (print-form-then-eval "(require '[speculoos.core :refer [only-invalid]])")]

     [:p "Now we can focus."]

     [:pre (print-form-then-eval "(only-invalid (validate-scalars {:version 1234 :date {:year 2055 :month \"Octoberfest\" :day 1} :person {:name \"Paul Bunyan\" :email \"babe@blue.ox\"}} {:version version-predicate :date date-spec :person person-spec}))")]

     [:p "Aha. One scalar datum failed to satisfy the predicate it was paired with. " [:code "\"Octoberfest\""] " is not a month enumerated by our month predicate."]

     [:p "So far, our changelog entry has a version number, a date, and a person. In the introduction, we outlined that a changelog entry would contain more info than that. So let's expand it."]

     [:p "It would be nice to tell people whether that release was breaking relative to the previous version. The initial release doesn't have a previous version, so it's breakage will be " [:code "nil"] ". For all subsequent versions, breakage will carry a " [:code "true"] " or " [:code "false"] " notion, so we'll require that datum be a boolean or " [:code "nil"] "."]

     [:pre (print-form-then-eval "(defn breaking-predicate [b] (or (nil? b) (boolean? b)))")]

     [:p "Also, it would be nice if we indicate the status of the project upon that release. A " [:a {:href "https://github.com/metosin/open-source/blob/main/project-status.md"} "reasonable enumeration of a project's status"] " might be " [:em "experimental"] ", "  [:em "active"] ", "  [:em "stable"] ", "  [:em "inactive"] ", or "  [:em "deprecated"] ". Since a valid status may only be one of a handful of values, a set makes a good membership predicate."]

     [:pre (print-form-then-eval "(def status-predicate #{:experimental :active :stable :inactive :deprecated})")]

     [:p "Let's assemble the version predicate, the breaking predicate, and the status predicate into another partial, temporary specification."]

     [:pre [:code "{:version version-predicate\n :breaking? breaking-predicate\n :project-status status-predicate}"]]

     [:p "Now that we have another temporary, partical specification, let's use it to validate some valid data (data in the upper row, specification in the lower row)."]

     [:pre (print-form-then-eval "(valid-scalars? {:version 99 :breaking? false :project-status :stable} {:version version-predicate :breaking? breaking-predicate :project-status status-predicate})" 115 25)]

     [:p "Now, let's validate some invalid data."]
     
     [:pre (print-form-then-eval "(valid-scalars? {:version 123 :breaking? true :project-status \"finished!\"} {:version version-predicate :breaking? breaking-predicate :project-status status-predicate})" 115 25)]

     [:p "Perhaps we're curious about exactly which datum failed to satisfy its predicate. So we switch to " [:code "validate-scalars"] " and filter with " [:code "only-invalid"] "."]

     [:pre (print-form-then-eval "(only-invalid (validate-scalars {:version 123 :breaking? true :project-status :finished!} {:version version-predicate :breaking? breaking-predicate :project-status status-predicate}))" 115 75)]

     [:p "Yup. Scalar " [:code ":finished!"] " is not enumerated by " [:code "status-predicate"] "."]

     [:p "A comment concerning a version is a free-form string, so we can use a bare " [:code "string?"] " predicate. Upgrade urgency could be represented by three discrete levels, so a set " [:code "#{:low :medium :high}"] " makes a fine predicate."]

     [:p "Now that we've got all the individual components for validating the version number, date (with year, month, day), person responsible (with name and email), project status, breakage, urgency, and a comment, we can assemble the specification for one changelog entry."]

     [:pre [:code "{:version version-predicate\n :date date-spec\n :responsible person-spec\n :project-status status-predicate\n :breaking? breaking-predicate\n :urgency #{:low :medium :high}\n :comment string?}"]]

     [:p "Let's use that specification to validate some data. Here's a peek behind the curtain: At this very moment, I don't have sample data to show you. I need to write some. I'm going to take advantage of the fact that a Speculoos specification is a regular Clojure data structure whose shape mimics the data. I already have the specification in hand. I'm going to copy-paste the specification, delete the predicates, and then insert some scalars."]

     [:p "Here's the specification with the predicates deleted."]
     
     [:pre [:code "{:version ___\n :date {:year ___\n        :month ___\n        :day ___}\n :responsible {:name ___\n               :email___}\n :project-status ___\n :breaking? ___\n :urgency ___\n :comment ___}"]]

     [:p "That will serve as a template. Then I'll insert some scalars."]

     [:pre [:code "{:version 55\n :date {:year 2025\n        :month \"December\"\n        :day 31}\n :responsible {:name \"Rowlf\"\n               :email \"piano@example.org\"}\n :project-status :active\n :breaking? false\n :urgency :medium\n :comment \"Performace improvements and bug fixes.\"}"]]

     [:p "Let's run a validation with that data and specification."]

     [:pre (print-form-then-eval "(valid-scalars?
                                  {:version 55
                                   :date {:year 2025
                                          :month \"December\"
                                          :day 31}
                                   :responsible {:name \"Rowlf Dog\"
                                                 :email \"piano@example.org\"}
                                   :project-status :active
                                   :breaking? false
                                   :urgency :medium
                                   :comment \"Performace improvements and bug fixes.\"}
                                  
                                  {:version version-predicate
                                   :date date-spec
                                   :responsible person-spec
                                   :project-status status-predicate
                                   :breaking? breaking-predicate
                                   :urgency #{:low :medium :high}
                                   :comment string?})")]

     [:p "Since I wrote the data based on the specification, it's a good thing the data is valid."]

     [:p "Let me change the version to a string, validate with the verbose " [:code "validate-scalars"] " and filter the output with " [:code "only-invalid"] " to keep only the invalid scalar+predicate pairs."]

     [:pre (print-form-then-eval "(only-invalid (validate-scalars
                                  {:version \"foo-bar-baz\"
                                   :date {:year 2025
                                          :month \"December\"
                                          :day 31}
                                   :responsible {:name \"Rowlf Dog\"
                                                 :email \"piano@example.org\"}
                                   :project-status :active
                                   :breaking? false
                                   :urgency :medium
                                   :comment \"Performace improvements and bug fixes.\"
                                   :changes []}
                                  
                                  {:version version-predicate
                                   :date date-spec
                                   :responsible person-spec
                                   :project-status status-predicate
                                   :breaking? breaking-predicate
                                   :urgency #{:low :medium :high}
                                   :comment string?}))")]

     [:p "Yup. String " [:code "\"foo-bar-baz\""] " is not a valid version number according to " [:code "version-predicate"] ". If I had made a typo while writing that changelog entry, before it got any further, validation would have informed me that I needed to fix that version number."]

     [:p "In the introduction, we mentioned that each version entry could contain a sequence of maps detailing the specific changes. That sequence is associated to " [:code ":changes"] ". Maybe you noticed I snuck that into the data in the last example. We didn't write any predicates for that key-val, so " [:code "validate-scalars"] " ignored it (Motto #3). We won't ignore it any longer."]

     [:p "The nesting depth is going to get messy, so let's put aside the parent changelog entry and zoom in on what a change might look like. Way back at the beginning, of this case study, we introduced this example."]

     [:pre [:code "{:description \"Addition function `+` now handles floating point decimal number types.\"\n :reference {:source \"Issue #78\"\n             :url \"https://example.com/issue/87\"}\n :change-type :relaxed-input-requirements\n :breaking? false\n :altered-functions ['+]\n :date {:year 2025\n        :month \"November\"\n        :day 8}\n :responsible {:name \"Fozzie Bear\"\n               :email \"fozzie@wocka-industries.com\"}}"]]

     [:p "This 'change' entry provides details about who changed what, when, and a reference to an issue-tracker. A single version may bundle multiple of these change entries."]

     [:p "I'll copy-paste the sample and delete the scalars."]

     [:pre [:code "{:description ___\n :reference {:source ___\n             :url ___}\n :change-type ___\n :breaking? ___\n :altered-functions []\n :date {:year ___\n        :month ___\n        :day ___}\n :responsible {:name ___\n               :email ___}}"]]

     [:p "That'll be a good template for a change detail."]

     [:p "We can start filling in the blanks because we already have specifications for " [:em "date"] ", " [:em "person"] ", and " [:em "breaking"] ". Similarly, a description is merely free-form text which can be validated with a simple " [:code "string?"] " predicate."]

     [:pre [:code "{:description string?\n :reference {:source ___\n             :url ___}\n :change-type ___\n :breaking? breaking-predicate\n :altered-functions []\n :date date-spec\n :responsible person-spec}"]]

     [:p "Now we can tackle filling in the remaiing blanks. The " [:em "reference"] " associates this change to a issue-tracker. The " [:code ":source"] " is a free-form string (i.e., \"GitHub Issue #27\", etc.), while " [:code ":url"] " points to a web-assessible resource. Let's require that a valid entry be a string that starts with \"https://\". We can demonstrate that regex."]

     [:pre
      (print-form-then-eval "(re-find #\"^https:\\/{2}[\\w\\/\\.]*\" \"https://example.com\")" 50 50)
      [:br]
      [:br]
      [:br]
      (print-form-then-eval "(re-find #\"^https:\\/{2}[\\w\\/\\.]*\" \"ht://example.com\")" 50 55)]

     [:p "The first example returns a match (truthy), while the second example is a malformed url and fails to find a match (falsey)."]

     [:p "Different issue trackers have different ways of referring to issues, so to accommodate that, we can include an optional " [:code ":ticket"] " entry that can be a free-form string or a " [:span.small-caps "uuid"] "."]

     [:pre (print-form-then-eval "(defn ticket-predicate [t] (or (string? t) (uuid? t)))")]

     [:p "Let's assemble those predicates to define this sub-component."]
     
     [:pre (print-form-then-eval "(def reference-spec {:source string?
                                                        :url #\"^https:\\/{2}[\\w\\/\\.]*\"
                                                        :ticket ticket-predicate})")]

     
     [:p "Our change specification currently looks like this."]
     
     [:pre [:code "{:description string?\n :reference reference-spec\n :change-type ___\n :breaking? breaking-predicate\n :altered-functions []\n :date date-spec\n :responsible person-spec}"]]

     
     [:pre (print-form-then-eval "(def change-kinds #{:initial-release

                                                       :security

                                                       :performance-improvement
                                                       :performance-regression

                                                       :memory-improvement
                                                       :memory-regression

                                                       :network-resource-improvement
                                                       :network-resource-regression

                                                       :added-dependency
                                                       :removed-dependency

                                                       :added-functions
                                                       :renamed-functions
                                                       :moved-functions
                                                       :removed-functions
                                                       :altered-functions

                                                       :function-arguments
                                                       :relaxed-input-requirements
                                                       :stricter-input-requirements

                                                       :increased-return
                                                       :decreased-return
                                                       :altered-return

                                                       :defaults

                                                       :implementation
                                                       :source-formatting
                                                       :error-messsage

                                                       :tests
                                                       :bug-fix
                                                       :deprecated-something

                                                       :policy
                                                       :meta-data
                                                       :documentation
                                                       :website
                                                       :release-note})")]


     ;; see Metosin's 'Project Status Model'
     ;; https://github.com/metosin/open-source/blob/main/project-status.md
     ;;   * experimental: Not recommended for production use.  No support nor maintenance. Testing and feedback welcome.
     ;;   * active: Actively developed. Recommended for use.
     ;;   * stable: Maintained and recommended for use. No major new features, but PRs are welcome.
     ;;   * inactive: Okay for production use. Will receive security fixes, but no new developments. Not recommended for new projects.
     ;;   * deprecated: Not recommended for any use.





     [:pre (print-form-then-eval "(def renamed-function-spec {:old-function-name symbol?
                                                               :new-function-name symbol?})")]

     [:pre (print-form-then-eval "(def change-scalar-spec {:date date-spec
                                                            :description string?
                                                            :reference reference-spec
                                                            :change-type change-kinds
                                                            :breaking? breaking-predicate
                                                            :added-functions (repeat symbol?)
                                                            :renamed-functions (repeat renamed-function-spec)
                                                            :moved-functions (repeat symbol?)
                                                            :altered-functions (repeat symbol?)
                                                            :removed-functions (repeat symbol?)})")]





     #_(comment
         
         (get-in* changelog-data [1 :changes 2])

         (only-invalid (validate-scalars (get-in* changelog-data [1 :changes 2])
                                         change-scalar-spec))

         (validate-scalars (get-in* changelog-data [1 :changes 2])
                           change-scalar-spec)

         )




     

     [:pre (print-form-then-eval "(def version-scalar-spec {:date date-spec
                                                             :responsible person-spec
                                                             :version version-predicate
                                                             :comment string?
                                                             :project-status status-predicate
                                                             :stable boolean?
                                                             :urgency #{:low :medium :high}
                                                             :breaking? boolean?})")]


     #_(comment
               (validate-scalars (get-in* changelog-data [2])
                                 version-scalar-spec)

               )


     "Composition: assembling the _version_ and _change_ specifications."

     "There's no rule that says we've got to do everything at once. Let's start small."

     [:pre (print-form-then-eval "(def changelog-scalar-spec [version-scalar-spec])")]

     [:pre (print-form-then-eval "(def changelog-scalar-spec [version-scalar-spec
                                                               version-scalar-spec])")]

     [:pre (print-form-then-eval "(def changelog-scalar-spec [(assoc version-scalar-spec :changes [change-scalar-spec])])")]

     [:pre (print-form-then-eval "(def changelog-scalar-spec [version-scalar-spec
                                                               version-scalar-spec
                                                               (assoc version-scalar-spec :changes [change-scalar-spec
                                                                                                    change-scalar-spec
                                                                                                    change-scalar-spec])])")]

     #_(comment
               (validate-scalars changelog-data
                                 changelog-scalar-spec)
               )


     [:pre (print-form-then-eval "(def changelog-scalar-spec (repeat (assoc version-scalar-spec :changes (repeat change-scalar-spec))))")]

     "Fun! A `clojure.lang.repeat` nested in a `clojure.lang.repeat`. Speculoos can handle that without a sweating. As long as there's not a repeat at the same path in the data. And there isn't. The changelog is hand-written, with each entry unique."

     #_[:pre (print-form-then-eval "(validate-scalars changelog-data
                                                     changelog-scalar-spec)")]

     #_[:pre (print-form-then-eval "(only-invalid (validate-scalars changelog-data
                                                                   changelog-scalar-spec))")]

     "Not in the least performant. But 1) Speculoos is experimental, exploring the concepts. No attention to performance, yet. 2) This style of 'off-line' validation (i.e., at the REPL, not in the middle of a high-throughput pipeline)...it's good enough."

     


     
     ] ;; end of scalar section

 ;; end of scalar validation sections

    
    #_[:section#collections
       [:h3 "Specifying & validating collections"]
       "No need to validate everything in the universe. Let's just do two. 1. Make sure we have required keys. 2. Verify relationship between version numbers."

       "1. Ensuring required keys"

       (def version-required-keys #{:date
                                    :responsible
                                    :version
                                    :comment
                                    :project-status
                                    :stable
                                    :urgency
                                    :breaking?
                                    :changes})

       (def changes-required-keys #{:description
                                    :date
                                    :change-type
                                    :breaking?})


       (defn contains-required-keys?
         "Returns a predicate that tests whether a map passed as the first argument
  contains all keys enumerated in set `req-keys`."
         {:UUIDv4 #uuid "71880b60-6ce7-4477-84f0-f8716b047692"}
         [req-keys]
         #(empty? (clojure.set/difference req-keys (set (keys %)))))


       (def version-coll-spec {:req-ver-keys? (contains-required-keys? version-required-keys)
                               :changes (vec (repeat 99 {:req-chng-keys? (contains-required-keys? changes-required-keys)}))})


       (comment
         (let [c1 (get-in* changelog-data [1 :changes 2])
               c2 (dissoc c1 :description)]
           (validate-collections c2
                                 {:req-chng-keys? (contains-required-keys? changes-required-keys)}))


         (validate-collections (get-in* changelog-data [2])
                               version-coll-spec)
         )

       (def changelog-coll-spec [version-coll-spec])

       (comment
         (validate-collections changelog-data
                               changelog-coll-spec)
         )

       (def changelog-coll-spec [version-coll-spec
                                 version-coll-spec])

       (comment
         (validate-collections changelog-data
                               changelog-coll-spec)
         )


       (def changelog-coll-spec (repeat version-coll-spec))

       (comment
         (only-invalid (validate-collections changelog-data
                                             changelog-coll-spec))
         )


       "2. Validating proper version incrementing"

       "Someone might reasonably point out that that manually declaring the version number _inside_ a sequential collection is redundant and error-prone. But, I may change my mind in the future and swith to dotted version numbers, version letters, or some other format. Plus, the changelog is intended to be manchine- _and_ human-readable (with priority on the latter), and the subsections are split between differnt files. So it's more ergonomic to put in an explicit version number."



       (defn properly-incrementing-versions?
         "Returns `true` if each successive version is exactly one more than previous."
         {:UUIDv4 #uuid "c937abfd-d230-4cd7-81c5-0f1a67ab911a"}
         [c-log]
         (every? #{1} (map #(- (:version %2) (:version %1)) c-log (next c-log))))


       (comment
         (def test-c-log-1 [{:version 0}
                            {:version 1}
                            {:version 2}
                            {:version 3}
                            {:version 4}])

         (def test-c-log-2 [{:version 0}
                            {:version 1}
                            {:version 99}])

         (properly-incrementing-versions? test-c-log-1)
         (properly-incrementing-versions? test-c-log-2)

         (every? #{1} (map #(- (:version %2) (:version %1)) test-c-log-1 (next test-c-log-1)))

         (validate-collections changelog-data
                               [properly-incrementing-versions?])
         )

       "The changelog collection specification is a vector --- mimicing the shape of the changelog data --- containing the `properly-incrementing-versions?` predicate followed by an infinite number of version collection specificaitons."

       (def changelog-coll-spec (conj [properly-incrementing-versions?] (repeat version-coll-spec)))

       (only-invalid (validate-collections changelog-data
                                           changelog-coll-spec))


       ] ;; end of collection section

    
    #_[:section#combo
       [:h3 "Combo validation"]
       "Finally, we can do a combo so that both scalars and collections are validated with a single function invocation."

       (only-invalid (validate changelog-data
                               changelog-scalar-spec
                               changelog-coll-spec))

       (valid? changelog-data
               changelog-scalar-spec
               changelog-coll-spec)
       ] ;; end of combo validaton section

    [:section
     [:p "Writing the specification for some data (or a function) is a legit strategy for dev. Kinda like writing unit tests first."]]
    
    ] ;; end of [:article]
   ] ;; end of [:body]
  )


#_(def page-body
    [:body
     [:article
      [:h1 "Case study: Specifying and validating a library changelog"]
      [:p "Foo bar baz."]
      [:p "Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat non proident, sunt in culpa qui officia deserunt mollit anim id est laborum."]]])


(def case-study-UUID #uuid "e3856cb2-b1d6-40cb-8659-8f2e7e56fcca")


(spit "doc/case_study.html"
      (page-template
       "Case study: Specifying and validating a library changelog"
       case-study-UUID
       page-body))
