(ns extended-extend-protocol.core
  "A single Clojure macro for streamlined protocol extension.")


(defmacro multi-extend-protocol
  "A version of `clojure.core/extend-protocol` with (possibly) more conciseness.

  Extends protocols to multiple types which share identical implementations. A
  series of names appearing before an implementation of the protocol expands
  into repeated implementations for each individual type.

  Recommended: Supply class/type names as symbols or `nil`; expressions intended
  to be evaluated will convey the inconsistent behavior of the underlying
  `clojure.core/extend-protocol`.

  Note: No additional checking is performed to ensure that the protocol's
  implementation is valid.

  Example:
  ```clojure
  (multi-extend-protocol AProtocol
    AType
    BType
    AClass
    (foo [x] x)
    (bar [y] y)
    (baz [z] z)

    DType
    EClass
    (foo [x] (inc x))
    (bar [y] (dec y))
    (baz [z] (identity z))

    FType
    nil
    (foo [_] 99)
    (bar [_] 42)
    (baz [_] 0))
  ```

  ...expands to...

  ```clojure
  (clojure.core/extend-protocol AProtocol
    AType
    (foo [x] x)
    (bar [y] y)
    (baz [z] z)

    BType
    (foo [x] x)
    (bar [y] y)
    (baz [z] z)

    AClass
    (foo [x] x)
    (bar [y] y)
    (baz [z] z)

    DType
    (foo [x] (inc x))
    (bar [y] (dec y))
    (baz [z] (identity z))

    EClass
    (foo [x] (inc x))
    (bar [y] (dec y))
    (baz [z] (identity z))

    FType
    (foo [_] 99)
    (bar [_] 42)
    (baz [_] 0)

    nil
    (foo [_] 99)
    (bar [_] 42)
    (baz [_] 0))
  ```"
  {:UUIDv4 #uuid "d734f711-ea0b-4388-8188-9b45ec510298"
   :author "Brad Losavio"
   :email "blosavio@sagevisuals.com"
   :org "com.sagevisuals"
   :license {:name "MIT"
             :url "https://opensource.org/license/MIT"}}
  [protocol & types+methods]
  (let [expanded (loop [partitioned (partition-by #(or (symbol? %) (nil? %)) types+methods)
                        acc []]
                   (if (seq partitioned)
                     (recur (nthrest partitioned 2)
                            (reduce #(apply conj %1 %2 (second partitioned)) acc (first partitioned)))
                     acc))]
    `(extend-protocol ~protocol ~@expanded)))

