(ns rpm-shared.aws.secrets
  (:require [clojure.data.json :as json]
            [clojure.string :as string]
            [cognitect.aws.client.api :as aws]
            [taoensso.encore :as enc]
            [rpm-shared.aws.client :as aws.client]
            [rpm-shared.aws.secrets :as aws.secrets]))

(defn secrets-manager [aws-user]
  (aws.client/client-memoized aws-user :secretsmanager))

(defn get-secret [aws-user secret-id]
  (let [secret (aws/invoke (secrets-manager aws-user)
                           {:op      :GetSecretValue
                            :request {:SecretId secret-id}})]
    (when (:SecretString secret)
      (assoc secret :value (json/read-str (:SecretString secret))))))

(defn values
  "Read `m`s values from the given secret and return a map with `m`s keys."
  ([aws-user secret-id m] (values (get-secret aws-user secret-id) m))
  ([secret m] (enc/map-vals #(get (:value secret) %) m)))

(defn list-secrets
  [aws-user]
  (aws.client/invoke (secrets-manager aws-user)
                     {:op :ListSecrets :request {}}
                     :SecretList))

(comment
  (aws.client/doc (secrets-manager :default) :ListSecrets)
  )

(comment
  (list-secrets :default)
  )

(defn update-secret!
  "Defaults to dry-run? true"
  ([aws-user secret-id new-value]
   (update-secret! aws-user secret-id new-value true))
  ([aws-user secret-id new-value dry-run?]
   (let [secret-string (cond
                         (map? new-value) (json/write-str new-value)
                         (string? new-value) new-value
                         :else (throw (ex-info (str "update-secret!: " secret-id
                                                    " Not a map, or string: "
                                                    new-value)
                                               {})))
         payload       {:op      :PutSecretValue
                        :request {:SecretId     secret-id
                                  :SecretString secret-string}}
         command       (if dry-run?
                         (fn [_ payload] (println (str "dry run: " payload)))
                         aws/invoke)]
     (command (secrets-manager aws-user) payload))))

(comment
  (aws.client/doc (secrets-manager :app) :PutSecretValue)
  )

(defn random-password [aws-user]
  (-> (aws/invoke (aws.secrets/secrets-manager aws-user)
                  {:op      :GetRandomPassword
                   :request {:PasswordLength          20
                             :ExcludeCharacters       "@\\/'\""
                             :ExcludeNumbers          false
                             :ExcludePunctuation      false
                             :ExcludeUppercase        false
                             :ExcludeLowercase        false
                             :IncludeSpace            false
                             :RequireEachIncludedType true}})
    :RandomPassword))

(comment
  (random-password :default)
  (aws.client/doc (secrets-manager :default) :GetRandomPassword)
  )

(def fix-private-key-format
  "AWS secrets manager replaced newlines with spaces when storing the private key
  Returns private key with newlines instead of spaces outside of header and footer."
  ;; five dashes, upper case letters and spaces, five dashes
  (let [header-footer-pattern #"-{5}+(\p{Lu}| )+-{5}+"]
    (fn [private-key]
      (let [matcher (re-matcher header-footer-pattern private-key)
            header  (do (re-find matcher) (subs private-key (.start matcher) (.end matcher)))
            start   (inc (.end matcher))
            footer  (do (re-find matcher) (subs private-key (.start matcher) (.end matcher)))
            end     (dec (.start matcher))
            body    (subs private-key start end)]
        (string/join "\n" [header (string/replace body " " "\n") footer])))))
