(ns rpm-shared.aws.ecr
  (:require [clojure.string :as string]
            [cognitect.aws.client.api :as aws]
            [rpm-shared.aws.client :as aws.client]))

(defn ecr [aws-user]
  (aws.client/client-memoized aws-user :ecr))

(defn list-images
  [aws-user repository-name tagged-only?]
  (aws.client/invoke (ecr aws-user)
                     {:op      :ListImages
                      :request (cond-> {:repositoryName repository-name}
                                 tagged-only? (assoc-in [:filter :tagStatus] "TAGGED"))}
                     :imageIds))

(comment
  (aws.client/doc (ecr :default) :ListImages)
  )

(defn describe-images
  [aws-user repository-name tagged-only?]
  (aws.client/invoke (ecr aws-user)
                     {:op      :DescribeImages
                      :request (cond-> {:repositoryName repository-name}
                                 tagged-only? (assoc-in [:filter :tagStatus] "TAGGED"))}
                     :imageDetails))

(comment
  (aws.client/doc (ecr :default) :DescribeImages)
  )

(defn describe-image [aws-user repository-name image-tag]
  (first (filter #((into #{} (:imageTags %)) image-tag)
                 (describe-images aws-user repository-name true))))

(defn batch-get-image
  [aws-user repository-name image-digest-or-tag]
  (let [image-id (if (string/starts-with? image-digest-or-tag "sha256:")
                   {:imageDigest image-digest-or-tag}
                   {:imageTag image-digest-or-tag})]
    (aws.client/invoke (ecr aws-user)
                       {:op      :BatchGetImage
                        :request {:repositoryName repository-name
                                  :imageIds       [image-id]}}
                       :images)))

(comment
  (aws.client/doc (ecr :default) :BatchGetImage)
  )

(defn tag-image!
  "Will remove `image-tag` from any image that had it before and assign it
  to the given image."
  [aws-user repository-name image-digest-or-tag image-tag]
  (let [image-manifest (-> (batch-get-image aws-user
                                            repository-name
                                            image-digest-or-tag)
                         first
                         :imageManifest)]
    (aws/invoke (ecr aws-user) {:op      :PutImage
                                :request {:repositoryName repository-name
                                          :imageManifest  image-manifest
                                          :imageTag       image-tag}})))

(comment
  (aws.client/doc (ecr :default) :PutImage)
  )
