(ns rpm-shared.aws.ec2
  (:require [rpm-shared.aws.client :as aws.client]))

(defn ec2 [aws-user] (aws.client/client-memoized aws-user :ec2))

(defn- make-filters
  "Convert {:filter-name :filter-value} or {:filter-name [:filter-value ...]} to
  the format expected by `ec2/describe-instances`."
  [m]
  (reduce-kv (fn [coll k v]
               (if (coll? v)
                 (conj coll {:Name (name k) :Values (mapv name v)})
                 (conj coll {:Name (name k) :Values [(name v)]})))
             []
             m))

(comment
  (make-filters {:a [:b :c]})
  (make-filters {:instance-state-name :running
                 :tag-key             :Foo})
  )

(defn tag-value
  "Extract a value from `:Tags` field as returned by `ec2/describe-instances`."
  [instance tag-key]
  (->> (:Tags instance)
    (filter #(= (:Key %) tag-key))
    first
    :Value))

(defn describe-instances
  "Syntactic sugar, see example below
  https://docs.aws.amazon.com/cli/latest/reference/ec2/describe-instances.html
  https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Filtering.html
  Returns list of instances"
  [aws-user filters]
  (mapcat :Instances
          (aws.client/invoke (ec2 aws-user)
                             {:op      :DescribeInstances
                              :request {:Filters (make-filters filters)}}
                             :Reservations)))

(comment ; list running instances, compare with aws console
  (describe-instances :default {:instance-state-name :running})
  (aws.client/doc (ec2 :default) :DescribeInstances)
  )

