(ns rpm-shared.aws.client
  "Helpers for using `cognitect.aws.client.api`"
  (:require [clojure.string :as string] 
            [cognitect.aws.client.api :as aws]
            [cognitect.aws.credentials :as creds]))

(defn credentials
  "First reads the environment variables `AWS_ACCESS_KEY_ID` and
  `AWS_SECRET_ACCESS_KEY`. Should that fail, reads from `~/.aws/credentials`.
  If you do not have access, check the IAM user groups you are in, or not in. 
  Returns a credentials provider."
  ([profile]
   (reify
     creds/CredentialsProvider
     (fetch [_]
       (creds/valid-credentials
         (if-let [creds (creds/fetch (creds/environment-credentials-provider))]
           creds
           (let [profile (cond
                           (string? profile) profile
                           (keyword? profile) (name profile)
                           (symbol? profile) (name profile)
                           :else (throw (ex-info (if (nil? profile)
                                                   "An aws-user is required"
                                                   (str "unknown aws-user: " profile)) {})))]
             (creds/fetch (creds/profile-credentials-provider profile)))))))))

(comment
  (.fetch (creds/environment-credentials-provider))
  (.fetch (credentials))
  (.fetch (credentials :app))
  (.fetch (credentials :default))
  (.fetch (credentials "default"))
  (.fetch (credentials (symbol "default")))
  )

(defn client
  "Defaults the default profile, region `eu-west-1`, and validating requests."
  ([api] (client :default api "eu-west-1"))
  ([profile api] (client profile api "eu-west-1"))
  ([profile api region] (client profile api region true))
  ([profile api region validate?]
   (doto
     (aws/client {:api                  api
                  :credentials-provider (credentials profile)
                  :region               region})
     (aws/validate-requests validate?))))

(def client-memoized (memoize client))

(defn invoke
  "Loops until NextToken is no longer returned. Throws on error."
  ([client op]
   (let [result (aws/invoke client op)]
     (if-let [error-response (:ErrorResponse result)]
       (throw (ex-info (:Message (:Error error-response)) 
                       (:Error error-response)))
       result)))
  ([client op result-key]
   (let [result (invoke client op)]
     (when-let [data (get result result-key)] ; nil return is ok
       (if-let [next-token (get result :nextToken)]
         (reduce conj
                 data
                 (invoke client
                         (assoc-in op [:request :nextToken] next-token)
                         result-key))
         data))))
  ([profile api op result-key] 
   (invoke (client-memoized profile api) op result-key))
  ([profile api region op result-key] 
   (invoke (client-memoized profile api region) op result-key))
  ([profile api region validate? op result-key]
   (invoke (client-memoized profile api region validate?) op result-key)))

(defn doc
  "Slightly more REPL friendly than aws/doc."
  [client operation]
  (when-let [docs (some-> client aws/ops operation)]
    (println (-> (:documentation docs)
               (string/split #"</p>")
               (string/replace #"\W*<p>\W*" "\n")
               (string/replace #"\. " ".\n")))
    (dissoc docs :documentation )))
