(ns rpm-shared.event-source
  "A general purpose, customizable, event generator.
  Please refer to `event-source` function doc string for more
  details."
  (:import [java.util.concurrent DelayQueue]))

(defn uuid-str
  "A small utility to return a string uuid."
  []
  (.toString (java.util.UUID/randomUUID)))

(defn chance [p] (< ^double (rand) (double p)))

(defn- parse-wait-spec
  "Returns a (fn [])-><msecs>
  event-spec is composed by:
  - an event-type for example :request
  - a probability to transition to next step
  - a wait-spec which indicates after how many milliseconds to generate the next step (if any)
  wait-spec can be:
  - a tuple containing a min-max wait time
  - a constant value
  - a function that will return next wait time"
  [x]
  (cond
    (fn?      x)        x  ; Arbitrary fn
    (integer? x) (fn [] x) ; <const-msecs>
    (vector?  x)           ; [<min-msecs> <max-msecs>]
    (let [[^long xmin ^long xmax] x]
      (assert (nat-int? xmin))
      (assert (nat-int? xmax))
      (assert (>= xmax xmin))
      (fn []
        (+ xmin (long (rand-int (- xmax xmin))))))))

(comment
  ((parse-wait-spec              300))
  ((parse-wait-spec        [100 300]))
  ((parse-wait-spec (constantly 500))))

(defn default-fill->viewable-wait
  "This function is meant to work as an example and as a `good`
  default to compute the transition time from fill to impression.

  We are assuming that 85% of the time there will be a transition in
  the range of 100 to 2000 milliseconds (expressed as a linear
  interpolation between min and max) and, for the remaining cases,
  we expect a maximum time of 15 minutes (900000 milliseconds) and the
  growth to be exponential.
  We created the formulas using this nice little tool:
  https://mycurvefit.com/"
  ([] (Math/round (double (default-fill->viewable-wait (rand)))))
  ([x]
   (let [x (double x)]
     (assert (<= x 1))
     (assert (>= x 0))
     (if (> x 0.85)
       (* 900000.0 (Math/pow x 21.23611))
       (+ 100.0 (* x 2533.333)))))
  )

(comment
  (default-fill->viewable-wait 0.75) ;; => 1999.99975
  (default-fill->viewable-wait 0.85) ;; => 2253.33305
  (default-fill->viewable-wait 0.95) ;; => 302815.8065376061
  (default-fill->viewable-wait)
  )

(deftype QueueEntry [^long timestamp data]
  java.util.concurrent.Delayed

  (getDelay [this time-unit]
    (.convert time-unit
      (- timestamp (System/currentTimeMillis))
      java.util.concurrent.TimeUnit/MILLISECONDS))

  (compareTo [this other-entry]
    (let [other-timestamp (.timestamp ^QueueEntry other-entry)]
      (if (< timestamp other-timestamp)
        -1
        (if (= timestamp other-timestamp)
          0
          1)))))

(defn get-next-time
  [delay-ms]
  (+ (System/currentTimeMillis) (long delay-ms)))

(comment
  ;; DelayQueue usage example

  (def dq (DelayQueue.))

  (.put ^DelayQueue dq (QueueEntry. (get-next-time 9000) {:foo :bar}))

  (when-let [entry (.poll ^DelayQueue dq)]
    (.data ^QueueEntry entry))
  )

(def default-event-spec
  [[:request      0.10 default-fill->viewable-wait]
   [:viewable     0.15 [  40   100]]
   [:impression   0.30 [2000 30000]]
   [:tracking-25  0.50 [2000 30000]]
   [:tracking-50  0.50 [2000 30000]]
   [:tracking-75  0.50 [2000 30000]]
   [:tracking-100 0    nil        ]])

(defn event-source
  "Returns a generator (fn []) that returns events of form
  [<flow-id> <event-id>]. Simulates a \"real\" event source like a set of users
  that are generating events:
    - Over time
    - Where each event is part of a logical ordered sequence/flow
    - Where the logical flow can be terminated at any time
  i.e. each flow is a subset [ev_1 ... ev_n] of [ev_1 ... ev_N], n <= N
  with events occuring over time.
  Takes an `event-spec` of form:
  [[<event-id-1> <probability-to-transition-to-event-id-2> <wait-spec>]
   [<event-id-2> <probability-to-transition-to-event-id-3> <wait-spec>]
   ...
   [<event-id-N> 0]] to define the logical ordered sequence/flow, the
  probability of transitioning between events, and the time to wait before
  transitioning.
  `wait-spec` can be of the form:
    - <const-msecs>
    - [<min-msecs> <max-msecs>]
    - (fn []) -> <msecs>
  NB: generator always returns an event immediately on call, meaning
  the consumer can control the rate of events by calling generator more
  or less frequently."
  [{:keys [flow-id-fn event-spec]
    :or
    {flow-id-fn (fn [] (uuid-str))
     event-spec default-event-spec
     }}]
  (let [;; Events which are ready for delivery on next call
        ready-events_ (DelayQueue.)] ; [[<flow-id> <ev-idx>] ...]
    (with-meta
      (fn get-flow-event []
        (let [[flow-id ev-idx]
              (if-let [entry (.poll ^DelayQueue ready-events_)]
                ;; Pre-existing flow-id already exists, return it:
                (.data ^QueueEntry entry)
                ;; No pre-existing flow-id exists, gen a new one:
                [(flow-id-fn) 0])
              [ev-id transition-p wait-spec]
              (nth event-spec ev-idx)]

          (when (chance transition-p) ; Schedule a transition to next ev-idx?
            (let [wait-msecs ; How long to wait before transitioning
                  (long
                    (let [wait-fn (parse-wait-spec wait-spec)]
                      (wait-fn #_flow-id #_ev-id)))]

              #_(println [:flow-id flow-id :waiting wait-msecs :-> ev-idx])

              (assert (nat-int? wait-msecs))
              ;; put the next event related to the current flow-id in the DelayQueue
              (let [next-ev-idx (inc (long ev-idx))]
                (assert (<= next-ev-idx (count event-spec)))
                (.put ^DelayQueue ready-events_
                  (QueueEntry. (get-next-time wait-msecs) [flow-id next-ev-idx])))
              ))
          [flow-id ev-id]))
      {:ready-events_ ready-events_})))

(comment
  (def my-ev-src (event-source {}))

  (require 'taoensso.encore)
  (taoensso.encore/qb 1e5 (my-ev-src)) ; 471.78

  (repeatedly 16 (fn [] (Thread/sleep 100) (my-ev-src)))

  #_(["8554a60b-b43f-4724-847d-71a7b02161dd" :request] ["04f40ccb-c478-42fe-a457-622e55a598db" :request] ["0726345f-c843-42bf-8e80-61a5f78b86bb" :request] ["ad0f3393-8f56-49c1-b031-09a59b38e51f" :request] ["a2c44fd0-a40d-4426-a999-bc9349a1816d" :request] ["3e2e5971-14c3-4b08-a423-d59a41ea262f" :request] ["0726345f-c843-42bf-8e80-61a5f78b86bb" :fill] ["90303b01-86ba-440f-8eba-ba8502bd4c52" :request] ["0726345f-c843-42bf-8e80-61a5f78b86bb" :viewable] ["9fbdf606-baac-4c03-8655-72e22826219d" :request] ["144a381d-95ef-463a-a1ee-7bee2d8fa25c" :request] ["9fbdf606-baac-4c03-8655-72e22826219d" :fill] ["144a381d-95ef-463a-a1ee-7bee2d8fa25c" :fill] ["ef4a53b9-b573-4073-acaf-cd2c94a374d5" :request] ["e432f3d6-3940-4ded-90c5-ffafc2260bba" :request] ["7f8f308c-6f2f-4b60-96de-04ee098c01ef" :request])


  (def my-ev-src-autoinc (event-source {:flow-id-fn (let [curr-idx_ (atom 0)]
                                                      (fn [] (swap! curr-idx_ inc)))}))
  (repeatedly 16 (fn [] (Thread/sleep 100) (my-ev-src-autoinc)))

  #_([1 :request] [2 :request] [3 :request] [4 :request] [5 :request] [6 :request] [7 :request] [8 :request] [9 :request] [10 :request] [11 :request] [12 :request] [13 :request] [13 :fill] [14 :request] [12 :fill])

  )
