(ns rpm-shared.utils
  "Misc general-purpose utils."
  (:require
   [clojure.string :as str]
   [clojure.tools.reader.edn :as edn]))

;;;; Config

(defn- throw-on-missing-sys-val! [requested-id]
  (throw
    (ex-info
      (str "The requested JVM system property or environment variable ("
           requested-id
           ") doesn't exist!")
      {:requested-sys-val-id requested-id})))

(defn get-sys-val
  "If the named JVM system property or environment variable exists,
  returns it as a string. Otherwise returns an optional default value
  (if provided), or throws.

  Useful as a way to prevent hard-coding config!"
  ([id        ] (get-sys-val id ::throw))
  ([id default]
   (or
     (System/getProperty id) ; JVM property string
     (System/getenv      id) ; Environment variable string
     (if (= default ::throw)
       (throw-on-missing-sys-val! id)
       default))))

(comment (get-sys-val "HTTP_PORT" "80"))

(defn read-sys-val
  "Like `get-sys-val` but will _read_ system properties
  or environment variables as edn strings.

  Can return any type, incl. keywords, longs, arbitrary
  data structures, etc.

  Useful as a way to prevent hard-coding config!"

  ([id        ] (read-sys-val id ::throw))
  ([id default]
   (if-let [s (get-sys-val id nil)]
     (edn/read-string s)
     (if (= default ::throw)
       (throw-on-missing-sys-val! id)
       default))))

(comment (read-sys-val "HTTP_PORT" 80))

;;;; Coercion

(defn get-int
  "Returns given input (number or string) coerced to a
  Long (Clojure's standard integer type), otherwise
  returns nil."
  [x]
  (cond
    (number? x) (long x)
    (string? x)
    (try
      (Long/parseLong x)
      (catch NumberFormatException _
        (try
          (long (Float/parseFloat x))
          (catch NumberFormatException _ nil))))))

(comment (mapv get-int [nil :foo 10 10.5 "10" "10.5"]))

(def ^:private splunk-key
  (memoize
    (fn [x]
      (if (keyword? x)
        (let [n  (name x)
              qn (if-let [ns (namespace x)]
                   (str ns "__" n)
                   n)]
          (str/replace qn "-" "_"))
        (str x)))))

(comment (mapv splunk-key [:foo/bar :foo-bar "baz"]))

(defn map->splunk-str
  "Given a map, returns a string using Splunk-style
  key-value pairs like \"key1=value1, key2=value2, key3=value3\".

  These kinds of key-value pairs are detected by Splunk and can
  be used for automatic filtering, etc."
  [m]
  (if (empty? m)
    ""
    (str
      (str/join ", "
        (map
          (fn [[k v]]
            (let [v
                  (cond
                    (and (number? v) (not (ratio? v))) v ; Support graphs, etc.
                    (nil? v) "nil"
                    :else (str "\"" v "\"")) ; "s support spaces in values, etc.
                  ]
              (str (splunk-key k) "=" v)))
          m))

      ;; Hack to keep Splunk from trying to interpret text after kvs as part
      ;; of the final val.
      ;; I.e. to prevent "foo=10, bar=20\n\nsomething-else" from being
      ;; interpreted by Splunk as {:foo "10", :bar "20\n\nsomethingelse"}.
      ;;
      ;; As an alternative, could instead just wrap all vals in ""'s?
      ;;
      ", eol=\".\"")))

(comment
  (mapv map->splunk-str [nil {} {:a "a" :b 2 5 "foo" :c nil :d false :foo-bar "bax"}])
  ;; ["" "" "a=\"a\", b=2, 5=\"foo\", c=nil, d=\"false\", foo_bar=\"bax\", eol=\".\""]
  )
