(ns rpm-shared.hll
  "A small Clojure utility wrapper for some of the
  Java `stream-lib` library's HyperLogLog+ features.
  Ref. https://github.com/addthis/stream-lib."
  {:current-maintainers "Peter, Damiano, Alex"}
  (:refer-clojure :exclude [merge])
  (:import
   [com.clearspring.analytics.stream.cardinality
    HyperLogLogPlus
    HyperLogLogPlus$Builder]))

(defn ^HyperLogLogPlus hll+
  "Returns a new stateful `HyperLogLogPlus` instance with the given
  precision parameters.

  Approximate cardinality error and HLL size for given `precision`:
    Precision  4:  error 21.0585%   0.02kb
    Precision  5:  error 16.0748%   0.032kb
    Precision  6:  error 11.8781%   0.052kb
    Precision  7:  error  8.0183%   0.096kb
    Precision  8:  error  5.7404%   0.181kb
    Precision  9:  error  4.3451%   0.353kb
    Precision 10:  error  3.7046%   0.693kb
    Precision 11:  error  2.8746%   1.377kb
    Precision 12:  error  1.908%    2.741kb ; Default
    Precision 13:  error  1.2265%   5.473kb
    Precision 14:  error  0.7323%  10.933kb
    Precision 15:  error  0.4851%  21.858kb
    Precision 16:  error  0.3483%  43.702kb
    Precision 17:  error  0.2217%  87.394kb
    Precision 18:  error  0.1594% 174.774kb
    Precision 19:  error  0.1016% 349.538kb"

  ([         ] (HyperLogLogPlus. 12))
  ([precision] (HyperLogLogPlus. (int precision)))
  ([precision sparse-precision]
   (HyperLogLogPlus. (int precision) (int sparse-precision))))

(comment (hll+))

(defn card
  "Returns (approximate) number of unique objects counted by the HLL."
  [^HyperLogLogPlus hll]
  (.cardinality hll))

(comment (card (hll+)))

(defn offer!
  "Modifies the given HLL to add the given object/s. Returns true iff
  the HLL's cardinality was affected."
  ([^HyperLogLogPlus hll obj       ] (.offer hll obj))
  ([^HyperLogLogPlus hll obj & more]
   (reduce
     (fn [acc in] (or (.offer hll in) acc))
     (.offer hll obj)
     more)))

(comment
  (offer! (hll+) "hello")
  (offer! (hll+) "hello" "hi!")
  (offer! (hll+) #{:foo :bar} #{:foo :bar})
  (let [h1 (hll+)]
    [(offer! h1 "hello")
     (offer! h1 "hi!")
     (offer! h1 "hello")
     (offer! h1 "foo")
     (offer! h1 "hello" "hi!")
     (offer! h1 "hello" "bar")]))

(defn insert!
  "Like `offer` but returns the given HLL."
  ([hll           ]                             hll)
  ([hll obj       ]       (offer! hll obj)      hll)
  ([hll obj & more] (apply offer! hll obj more) hll))

(comment (card (insert! (hll+) #{:a :b} #{:a :b})))

(defn hll+->bytes
  "Returns given HLL serialized as a byte array."
  ^bytes [^HyperLogLogPlus hll]
  (.getBytes hll))

(defn bytes->hll+
  "Returns HLL deserialized from given byte array."
  ^HyperLogLogPlus [ba]
  (HyperLogLogPlus$Builder/build ^bytes ba))

(comment (count (seq (hll+->bytes (insert! (hll+) "foo")))))

(defn merge
  "Returns a new HLL created by merging the given HLLs.
  None of the given HLLs are modified."
  ([] nil)
  ([^HyperLogLogPlus hll       ]         hll)
  ([^HyperLogLogPlus hll & more] (.merge hll (into-array more))))

(comment
  (card
    (merge
      (insert! (hll+) "foo" "bar")
      (insert! (hll+) "foo" "baz"))))
