(ns rpm-shared.utils
  "Misc general-purpose utils."
  (:require
   [clojure.string :as str]
   [clojure.tools.reader.edn :as edn]))

;;;; Config

(defn- throw-on-missing-sys-val! [requested-id]
  (throw
    (ex-info
      "The requested JVM system property or environment variable doesn't exist!"
      {:requested-sys-val-id requested-id})))

(defn get-sys-val
  "If the named JVM system property or environment variable exists,
  returns it as a string. Otherwise returns an optional default value
  (if provided), or throws.

  Useful as a way to prevent hard-coding config!"
  ([id        ] (get-sys-val id ::throw))
  ([id default]
   (or
     (System/getProperty id) ; JVM property string
     (System/getenv      id) ; Environment variable string
     (if (= default ::throw)
       (throw-on-missing-sys-val! id)
       default))))

(comment (get-sys-val "HTTP_PORT" "80"))

(defn read-sys-val
  "Like `get-sys-val` but will _read_ system properties
  or environment variables as edn strings.

  Can return any type, incl. keywords, longs, arbitrary
  data structures, etc.

  Useful as a way to prevent hard-coding config!"

  ([id        ] (read-sys-val id ::throw))
  ([id default]
   (if-let [s (get-sys-val id nil)]
     (edn/read-string s)
     (if (= default ::throw)
       (throw-on-missing-sys-val! id)
       default))))

(comment (read-sys-val "HTTP_PORT" 80))

;;;; Coercion

(defn get-int
  "Returns given input (number or string) coerced to a
  Long (Clojure's standard integer type), otherwise
  returns nil."
  [x]
  (cond
    (number? x) (long x)
    (string? x)
    (try
      (Long/parseLong x)
      (catch NumberFormatException _
        (try
          (long (Float/parseFloat x))
          (catch NumberFormatException _ nil))))))

(comment (mapv get-int [nil :foo 10 10.5 "10" "10.5"]))
