 /*
  * The contents of this file are subject to the Mozilla Public
  * License Version 1.1 (the "License"); you may not use this file
  * except in compliance with the License. You may obtain a copy of
  * the License at http://www.mozilla.org/MPL/
  *
  * Software distributed under the License is distributed on an "AS
  * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
  * implied. See the License for the specific language governing
  * rights and limitations under the License.
  *
  * The Original Code is the Bugzilla Testopia Java API.
  *
  * The Initial Developer of the Original Code is Andrew Nelson.
  * Portions created by Andrew Nelson are Copyright (C) 2006
  * Novell. All Rights Reserved.
  *
  * Contributor(s): Andrew Nelson <anelson@novell.com>
  *					Jason Sabin <jsabin@novell.com>
  */
package tcms.API;

import java.util.HashMap;
import java.util.Map;

import org.apache.xmlrpc.XmlRpcException;

import com.redhat.qe.xmlrpc.Session;

/**
 * 
 * @author anelson, bstice 
 * Creates a wrapper class for the TestRunCase
 */
public class TestCaseRun extends TestopiaObject{
	public enum Statuses {IDLE, PASSED, FAILED, RUNNING, PAUSED, BLOCKED, ERROR};
			
	//stores the updated value until it's pushed to testopia with an update
	private StringAttribute notes = this.newStringAttribute("notes", null);
	private IntegerAttribute status= this.newIntegerAttribute("case_run_status", null);
	private IntegerAttribute assigneeID= this.newIntegerAttribute("assignee", null);

	private IntegerAttribute caseID= this.newIntegerAttribute("case", null);
	private IntegerAttribute runID= this.newIntegerAttribute("run", null);
	private IntegerAttribute buildID= this.newIntegerAttribute("build", null);
	private IntegerAttribute environmentID= this.newIntegerAttribute("environment", null);

	private Map<Statuses, Integer> statuses = new HashMap<Statuses, Integer>();
	

	/**
	 * Use this constructor if you just want to use gets
	 * @param userName your bugzilla username
	 * @param password your bugzilla password
	 * @param caseRunID ID generated by bugzilla - can be null
	 * @param url URL - the url of the testopia server that you want to connect to
	 */
	public TestCaseRun(Session session, Integer caseRunID)
	{
		this.session   = session; 
		this.id = newIntegerAttribute("case_run_id", caseRunID);
	}
	
	/**
	 * Use this constructor if you want to do sets and gets
	 * @param userName your bugzilla username
	 * @param password your bugzilla password 
	 * @param caseID ID used to get the case
	 * @param runID test run number
	 * @param buildID ID generated by bugzilla
	 * @param environmentID ID generated by bugzilla
	 * @param id ID generated by bugzilla - can be null
	 * @param url URL - the url of the testopia server that you want to connect to
	 */
	public TestCaseRun(Session session,	int runID,  int caseID, int buildID, int environmentID) 
	{
		this.session = session;
		this.caseID.set(caseID);
		this.runID.set(runID); 
		this.buildID.set(buildID); 
		this.environmentID.set(environmentID);
		this.id = newIntegerAttribute("case_run_id", null);

	}
	
	public TestCaseRun(Session session,	int runID,  int caseID, int buildID) 
	{
		this.session = session;
		this.caseID.set(caseID);
		this.runID.set(runID); 
		this.buildID.set(buildID); 
		this.id = newIntegerAttribute("case_run_id", null);
	}
	
	/**
	 * Updates are not called when the .set is used. You must call update after all your sets
	 * to push the changes over to testopia.
	 * @throws TestopiaException if planID is null 
	 * @throws XmlRpcException
	 * (you made the TestCase with a null caseID and have not created a new test plan)
	 */
	public Map<String,Object> update() throws TestopiaException, XmlRpcException
	{
		if (runID.get() == null) 
			throw new TestopiaException("runID is null.");
		//update the testRunCase
		return super.updateById("TestCaseRun.update");
	}
	
	/**
	 * Calls the create method with the attributes as-is (as set via constructors
	 * or setters).  
	 * @return a map of the newly created object
	 * @throws XmlRpcException
	 */
	public Map<String,Object> create() throws XmlRpcException{
		Map<String, Object> retval = super.create("TestCaseRun.create");
		this.setRunID((Integer)retval.get("case_run_id"));
		return retval;			
	}
	

	
	/**
	 * 
	 * @return a hashMap of all the values found. Returns null if there is an error
	 * and the TestCaseRun cannot be returned
	 * @throws XmlRpcException 
	 */
	@SuppressWarnings("unchecked")
	public Map<String, Object> getAttributes()
	throws XmlRpcException
	{
		Map m = (Map<String, Object>)callXmlrpcMethod("TestCaseRun.get", id);
		syncAttributes(m);
		return m;
		
	}

	/**
	 * This is used to append a note
	 * @param notes string - the note you want entered into the testCaseRun
	 */
	public void setNotes(String notes)
	{	
		this.notes.set(notes);
	}
	
	/**
	 * This is used to change the testCaseRun status (2 for pass, 3 for fail)
	 * @param status int - the status you want to change the testCaseRun to
	 * @throws XmlRpcException 
	 */
	public void setStatus(Statuses status) 
	{
		Integer iStat = statuses.get(status);
		if (iStat == null) {
			Map map;
			try {
				map = (Map) callXmlrpcMethod("TestCaseRun.check_case_run_status", status.toString().toLowerCase());
			}catch(XmlRpcException xe) {
				throw new TestopiaException("Could not retrieve status id for " + status, xe);
			}
			iStat = (Integer)map.get("id");
			statuses.put(status, iStat); //cache the value so we don't have to keep asking the remote end
		}
		this.status.set(iStat);
	}
	

	
	
	/**
	 * Changes the assigneeID of the testCaseRun
	 * @param assigneeID
	 */
	public void setAssigneeID(int assigneeID)
	{
		this.assigneeID.set(assigneeID); 
	}
	
	/**
	 * @return the caseID
	 */
	public int getCaseID() {
		return caseID.get();
	}

	/**
	 * @return the runID
	 */
	public int getRunID() {
		return runID.get();
	}

	/**
	 * @return the buildID
	 */
	public int getBuildID() {
		return buildID.get();
	}

	/**
	 * @return the environmentID
	 */
	public int getEnvironmentID() {
		return environmentID.get();
	}


	/**
	 * @return the notes
	 */
	public String getNotes() {
		return notes.get();
	}

	/**
	 * @return the assigneeID
	 */
	public int getAssigneeID() {
		return assigneeID.get();
	}
	
	public void setCaseID(Integer caseID) {
		this.caseID.set(caseID);
	}

	public void setRunID(Integer runID) {
		this.runID.set(runID);
	}

	public void setBuildID(Integer buildID) {
		this.buildID.set(buildID);
	}

	public void setEnvironmentID(Integer environmentID) {
		this.environmentID.set(environmentID);
	}

}
