(ns metricsaurus-rex.ring.middleware
  (:require [metricsaurus-rex.metrics :as metrics]
            [metricsaurus-rex.ring.util :as util])
  (:import [com.codahale.metrics MetricRegistry]
           [java.util.concurrent TimeUnit]))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; All routes middleware

(defn wrap-request-time
  "Ring middleware that times the total time of all requests.
   `timer-name` must be a sequence of 3, 4, or 5 strings."
  [handler timer-name]
  (let [timer (metrics/timer (metrics/metric-name timer-name))]
    (fn [request]
      (.time timer (partial handler request)))))

(defn wrap-request-meter [handler meter-name]
  (let [meter (metrics/meter (metrics/metric-name (conj meter-name "requests")))]
    (fn [request]
      (.mark meter)
      (handler request))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Per-route middleware

(defn wrap-per-route-timer
  "Middleware for per-route timing.
   Params:
    - handler - the ring handler to wrap
    - group - String, the top-level part of the MetricName
    - route-name-fn - function that takes the ring request and returns
                      the \"type\" part of the metric name, or nil if
                      the route should not be timed.
  "
  [handler group route-name-fn]
  (fn [request]
    (if-let [route-name (route-name-fn request)]
      (let [verb          (util/get-verb request)
            timer-name    (metrics/metric-name [group route-name verb])
            timer         (metrics/timer timer-name)
            timer-context (.time timer)]
        (.time timer (partial handler request)))
      (handler request))))


(defn- wrap-per-route-metric
  "Common implementation for wrap-per-route-[meter|counter]"
  [handler group route-name-fn mark-fn]
  (fn [request]
    (if-let [route-name (route-name-fn request)]
      (let [verb (util/get-verb request)]
        (try
          (let [[response status e] (try
                                      (let [response (handler request)]
                                        [response (:status response) nil])
                                      (catch Throwable e
                                        [nil 500 e]))
                response-group      (util/response-group status)
                metric-name         (metrics/metric-name [group route-name response-group verb])]
            (mark-fn metric-name)
            (if e
              (throw e)
              response))))
      (handler request))))

(defn wrap-per-route-meter
  "Middleware for per-route metering.
   Params:
    - handler - the ring handler to wrap
    - group - String, the top-level part of the MetricName
    - route-name-fn - function that takes the ring request and returns
                      the \"type\" part of the metric name, or nil if
                      the route should not be metered.
  "
  [handler group route-name-fn]
  (wrap-per-route-metric handler group route-name-fn
                         (fn [^String metric-name]
                           (metrics/mark! (metrics/metric-name metric-name "responses")))))


(defn wrap-per-route-counter
  "Middleware for per-route request-counting.
   Params:
    - handler - the ring handler to wrap
    - group - String, the top-level part of the MetricName
    - route-name-fn - function that takes the ring request and returns
                      the \"type\" part of the metric name, or nil if
                      the route should not be counted.
  "
  [handler group route-name-fn]
  (wrap-per-route-metric handler group route-name-fn (partial metrics/inc-counter!)))
