(ns metricsaurus-rex.metrics.snapshot
  "Functions for getting a snapshot of all metrics."
  (:require [clojure.string :as string]
            [metricsaurus-rex.metrics :as metrics])
  (:import [com.codahale.metrics MetricRegistry
            Clock Counter Meter Timer Gauge Histogram
            Metered Sampling Snapshot JvmAttributeGaugeSet]
           [java.util.concurrent TimeUnit]))

(defprotocol MetricMap
  (as-map [metric] "Render the Metric as a clojure map."))

(defn- metered-fields [^Metered m]
  {:count (.getCount m)
   :mean  (.getMeanRate m)
   :m1    (.getOneMinuteRate m)
   :m5    (.getFiveMinuteRate m)
   :m15   (.getFifteenMinuteRate m)})

(defn- sampling-fields [^Sampling sampling]
  (let [^Snapshot s (.getSnapshot sampling)]
    {:median (.getMedian s)
     :p75    (.get75thPercentile s)
     :p95    (.get95thPercentile s)
     :p98    (.get98thPercentile s)
     :p99    (.get99thPercentile s)
     :p999   (.get999thPercentile s)}))

(extend-protocol MetricMap
  Counter
  (as-map [c]
    {:type  "counter"
     :count (.getCount c)})

  Meter
  (as-map [m]
    (merge {:type "meter"}
           (metered-fields m)))

  Timer
  (as-map [t]
    {:type "timer"
     :duration (sampling-fields t)})

  Gauge
  (as-map [g]
    {:type "gauge"
     :value (try (.getValue g)
                 (catch RuntimeException e
                   (str "error reading gauge: " (.getMessage e))))})

  Histogram
  (as-map [h]
    (merge {:type  "histogram"
            :count (.getCount h)}
           (sampling-fields h))))

(defn- gather-metrics [metrics [name metric]]
  (assoc metrics name (as-map metric)))

(defn all-metrics []
  {:gauges     (reduce gather-metrics {} (.getGauges metrics/*registry*))
   :counters   (reduce gather-metrics {} (.getCounters metrics/*registry*))
   :histograms (reduce gather-metrics {} (.getHistograms metrics/*registry*))
   :meters     (reduce gather-metrics {} (.getMeters metrics/*registry*))
   :timers     (reduce gather-metrics {} (.getTimers metrics/*registry*))})
