(ns metricsaurus-rex.metrics
  (:require [clojure.string :as string]
            [robert.hooke :refer [add-hook clear-hooks]])
  (:refer-clojure :exclude [time])
  (:import [clojure.lang Var]
           [com.codahale.metrics MetricRegistry Counter Histogram Meter Timer Gauge JmxReporter CachedGauge RatioGauge RatioGauge$Ratio MetricFilter]
           [com.codahale.metrics.jvm  GarbageCollectorMetricSet MemoryUsageGaugeSet ThreadStatesGaugeSet]
           [java.util Set]
           [java.util.concurrent TimeUnit]))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; AsMetricName protocol.
;;; Turn (almost) anything into a MetricName.

(defprotocol AsMetricName
  (as-metric-name [name] "Convert the value into a MetricName instance."))

(defn- ^String ->s [o]
  (when o
    ((if (instance? clojure.lang.Named o) name str) o)))

(defn- ^String clean
  "Remove illegal characters from a metric name"
  [o]
  (-> o ->s (string/replace #"[^\.a-zA-Z0-9_-]" "")))

(defn- var-name-parts [^Var v]
  (-> v .ns .name name (string/split #"\.") vec (conj (-> v .sym name))))

(extend-protocol AsMetricName

  String
  (as-metric-name [s]
    (clean s))

  clojure.lang.Var
  (as-metric-name [v]
    (as-metric-name (var-name-parts v)))

  clojure.lang.Named
  (as-metric-name [o]
    (as-metric-name
     (if-let [ns (.getNamespace o)]
       (str ns \. (.getName o))
       (.getName o))))

  clojure.lang.Namespace
  (as-metric-name [ns]
    (as-metric-name (.name ns)))

  clojure.lang.Sequential
  (as-metric-name [[name & parts]]
    (let [names (map as-metric-name parts)]
      (MetricRegistry/name (as-metric-name name) (into-array String names)))))

(defn ^String metric-name [name & names]
  ;; this is here just because I can't figure out how to type-hint a
  ;; protocol-method properly
  (as-metric-name (cons name names)))

(defn ^MetricRegistry create-registry []
  (MetricRegistry.))

(defn metrics-defined? [^MetricRegistry registry]
  (let [^Set names (.getNames registry)]
    (not (.isEmpty names))))

(defn add-jvm-metrics [metric-registry]
  (doto metric-registry
    (.registerAll (GarbageCollectorMetricSet.))
    (.registerAll (MemoryUsageGaugeSet.))
    (.registerAll (ThreadStatesGaugeSet.))))

(defn- report-metrics-to-jmx [registry]
  (-> (JmxReporter/forRegistry registry)
      (.build)
      (.start)))

(defn create-default-registry []
  (doto (create-registry)
    (add-jvm-metrics)
    (report-metrics-to-jmx)))

(def ^:dynamic *registry*
  (create-default-registry))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Factory functions

(defn ^Timer timer [name]
  (.timer *registry* (metric-name name)))

(defn ^Meter meter [name]
  (.meter *registry* (metric-name name)))

(defn ^Counter counter [name]
  (.counter *registry* (metric-name name)))

(defn ^Histogram histogram [name]
  (.histogram *registry* (metric-name name)))

(defn- create-simple-gauge [f]
  (reify
    Gauge (getValue [this] (f))
    clojure.lang.IDeref (deref [this] {:value (f)})))

(defn- create-cached-gauge [f timeout timeout-unit]
  (proxy [CachedGauge clojure.lang.IDeref] [timeout timeout-unit]
    (loadValue []
      (f))
    (deref [] {:value (.getValue this)})))

(defn- register-gauge [name gauge]
  (let [name (metric-name name)]
    (.remove *registry* name)
    (.register *registry* name gauge)))

(defn gauge
  ([name f]
     (->> (create-simple-gauge f)
          (register-gauge name)))
  ([name f timeout timeout-unit]
     (->> (create-cached-gauge f timeout timeout-unit)
          (register-gauge name))))

(defn- create-ratio-gauge [of to]
  (let [metrics  (.getMetrics *registry*)
        of-meter (get metrics (metric-name of))
        to-meter (get metrics (metric-name to))]
    (proxy [RatioGauge] []
      (getRatio []
        (RatioGauge$Ratio/of (.getOneMinuteRate of-meter)
                             (.getOneMinuteRate to-meter))))))

(defn ratio [of to name]
  (let [ratio-gauge (create-ratio-gauge of to)]
    (register-gauge name ratio-gauge)
    ratio-gauge))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Functions for doing stuff

(defn time!
  "Time function execution."
  [name f]
  (.time (timer name) f))

(defn update-timer!
  "Update a timer with a specified interval."
  [name time-delta]
  (.update (timer name) time-delta TimeUnit/MILLISECONDS))

(defn mark!
  "Mark the meter."
  ([name]
     (mark! name 1))
  ([name n]
     (.mark (meter name) n)))

(defn inc-counter!
  "Increment the counter named by `name`."
  ([name]
     (inc-counter! name 1))
  ([name n]
     (.inc (counter name) n)))

(defn dec-counter!
  "Decrement the counter named by `name`."
  ([name]
     (dec-counter! name 1))
  ([name n]
     (.dec (counter name) n)))

(defn update-histogram!
  "Updates a histogram with the specified value."
  [name value]
  (.update (histogram name) (long value)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Functions for reading values of metrics

(defn counter-value
  "Return the current value of the counter named by `name`."
  [name]
  (.count (counter name)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Macros

(defmacro with-timer "Times execution of a body for a given timer-name (timer-name is defined by the key in the timers map)"
  [name & body]
  `(time! ~name (fn [] ~@body)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Functions for varjacking

(defn- metric-name+
  "Metric name with postfix:
   (metric-name+ #'foo.bar/qux \"-floop\") => (metric-name :foo.bar/qux-floop)
  "
  [^Var v postfix]
  (let [parts (var-name-parts v)
        name  (peek parts)]
    (metric-name (conj (pop parts) (str name postfix)))))

(defn time-var!
  "Inject timing around the function stored in the Var."
  [^Var v]
  (let [^Timer timer (timer (metric-name+ v "-timer"))]
    (add-hook v ::timer (fn [f & args]
                          (.time timer #(apply f args))))
    (alter-meta! v assoc-in [::metrics ::timer] timer)))

(defn meter-var!
  "Inject a meter into the var."
  [^Var v]
  (let [^Meter meter (meter (metric-name+ v "-meter"))]
    (add-hook v ::meter (fn [f & args]
                          (.mark meter)
                          (apply f args)))
    (alter-meta! v assoc-in [::metrics ::meter] meter)))

(defn dynamic-meter!
  "Meter the function, using a meter name that depends on the return value."
  [^Var v meter-name-fn]
  (add-hook v ::meter (fn [f & args]
                        (let [ret        (apply f args)
                              meter-name (meter-name-fn ret)]
                          (when meter-name
                            (.mark (meter meter-name)))
                          ret))))

(defn count-var!
  "Inject a counter that increments by one each time the wrapped var is called."
  [^Var v]
  (let [^Counter counter (counter (metric-name+ v "-counter"))]
    (add-hook v ::counter (fn [f & args]
                            (.inc counter)
                            (apply f args)))
    (alter-meta! v assoc-in [::metrics ::counter] counter)))

(defn var-timer
  "Get the Timer that has been injected into the Var with `time-var!`."
  [^Var v]
  (-> v meta ::metrics ::timer))

(defn var-meter
  "Get the Meter that has been injected into the Var with `meter-var!`."
  [^Var v]
  (-> v meta ::metrics ::meter))

(defn var-counter
  "Get the Counter that has been injected into the Var with `count-var!`."
  [^Var v]
  (-> v meta ::metrics ::counter))

(defn unjack
  "Remove all metrics from the given Var."
  [^Var v]
  (clear-hooks v)
  (alter-meta! v dissoc ::metrics))
