(ns clj-chef.api
  (:require [clj-chef.util :as util]
            [clj-http.client :as client]
            [cheshire.core :as json]
            [signet.ring :as signet]))

(def chef-version (util/config-property "CHEF_VERSION"))
(def user-id (util/config-property "CHEF_USER"))
(def host (util/config-property "CHEF_API_URL"))
(def org (util/config-property "CHEF_ORG"))
(def pem-location (util/config-property "CHEF_USER_PEM_LOCATION"))

(declare decoded-pem load-pem seal-headers)

(defn http-get
  ([path]
    (http-get path {}))
  ([path params]
    (let [sealed-headers (seal-headers (str org path) "" :GET user-id)]
      (-> (str host org path)
        (client/get (assoc sealed-headers :query-params params))
        (:body )
        (json/parse-string)))))

(defn list-environments []
  (http-get "/environments"))

(defn list-cookbooks []
  (http-get "/cookbooks"))

(defn list-clients []
  (http-get "/clients"))

(defn list-users []
  (http-get "/user"))

(defn seal-headers [path body request-method user-id]
  (-> (signet/seal {:uri path :body "json" :request-method request-method} user-id (decoded-pem))
    (assoc-in [:headers "X-Chef-Version"] chef-version)
    (assoc :host host)
    (assoc :throw-exceptions false)
    (assoc :accept "application/json")))

(defn decoded-pem []
  (signet/rsa-key (signet/pem-decode (.getBytes (load-pem)))))

(defn load-pem []
  (slurp pem-location))