(ns schema.extensions.derived
  (:require [schema.core :as s]
            [schema.spec.core :as spec]
            [schema.coerce :as coerce]
            [schema.utils :refer [error?]]
            #?(:clj [schema.macros :as sm])
            [schema.extensions.util :refer [field-meta]])
  #?(:cljs (:require-macros [schema.macros :as sm])))


;;; Derived
;;; schema elements derived on other schema elements at the same level

(defn derived
  "Calculate a field's value based on its parent object. Parameters are parent object and key name."
  [calculator]
  {:_derived {:calculator calculator}})

(defn defaulting
  "Allow a field to be a default. Is not applied if a value exists. Takes a function or a value."
  [default]
  (derived (fn [a k] (cond (some? (a k)) (a k)
                           (fn? default) (default)
                           :otherwise    default))))

(defn massaging
  "Calculate a fields value based on its current value"
  [massager]
  (derived (fn [a k] (massager (get a k)))))

;; Helpers

(defn derived? [schema]
  (-> schema field-meta :_derived some?))

(defn- derived-calculator [schema]
  (-> schema field-meta :_derived :calculator))

(defn derive-coercer [schema]
  (let [kvs (and (map? schema)
                 (filter (fn [[k v]] (derived? v)) schema))]
    (when (and kvs (not (empty? kvs)))
      (fn [data]
        (if-not (map? data) data
                (->> kvs
                     (map
                      (fn [[k ds]]
                        (try [k ((derived-calculator ds) data k)]
                             (catch #?(:clj Exception
                                       :cljs js/Error) e
                               [k (sm/validation-error ds data 'deriving-exception e)]))))
                     (apply concat)
                     (apply assoc data)))))))

(defn- deriver [input-schema]
  (spec/run-checker
   (fn [schema params]
     (if (some? schema)
       (let [checker (spec/checker (s/spec schema) params)
             drv (or (derive-coercer schema) identity)
             unjson (or (coerce/json-coercion-matcher schema) identity)]
         (fn [data]
           ;; TODO: unjson is 2nd because it was done that way before.
           ;;       seems like it should be first, though.  -JR
           (-> data drv unjson checker)))
       schema))
   true
   input-schema))


;; Deriving walker

(defn derive-schema [schema data]
  "Walk a schema, deriving any keys that are of the Derived type"
  ((deriver schema) data))
