(ns schema.extensions.derived
  (:require [schema.core :refer [subschema-walker explain start-walker walker] :as s]
            [schema.coerce :as coerce]
            [schema.utils :refer [error?]]
            #?(:clj [schema.macros :as sm])
            [schema.extensions.util :refer [field-update field-meta]])
  #?(:cljs (:require-macros [schema.macros :as sm])))


;;; Derived
;;; schema elements derived on other schema elements at the same level

(defn derived
  "Calculate a field's value based on its parent object. Parameters are parent object and key name."
  [schema calculator]
  (field-update schema {:_derived {:calculator calculator}}))

(defn defaulting
  "Allow a field to be a default. Is not applied if a value exists. Takes a function or a value."
  [schema default]
  (derived schema (fn [a k] (cond (some? (a k)) (a k)
                                  (fn? default) (default)
                                  :otherwise    default))))

(defn massaging
  "Calculate a fields value based on its current value"
  [schema massager]
  (derived schema (fn [a k] (massager (get a k)))))

(defn auto-defaulting
  [schema]
  (defaulting schema (empty schema)))

(defn optional [schema]
  (-> schema
      s/maybe
      (defaulting nil)))

;; Helpers

(defn derived? [schema]
  (-> schema field-meta :_derived some?))

(defn- derived-calculator [schema]
  (-> schema field-meta :_derived :calculator))

(defn derive-coercion [schema]
  (let [kvs (and (map? schema)
                 (filter (fn [[k v]] (derived? v)) schema))]
    (when (and kvs (not (empty? kvs)))
          (fn [data]
            (if-not (map? data) data
              (->> kvs
                   (map
                    (fn [[k ds]]
                      (try [k ((derived-calculator ds) data k)]
                           (catch #?(:clj Exception
                                     :cljs js/Error) e
                          [k (sm/validation-error ds data 'deriving-exception e)]))))
                   (apply concat)
                   (apply assoc data)))))))

(defn- ignore-walker-errors
  [walk]
  (fn [data]
    (let [result (walk data)]
      (if (error? result)
        (if (= data :schema.core/missing)
          nil
          data)
        result))))

(defn- json-coercer [schema]
  (start-walker
   (fn [subschema]
     (let [walk (ignore-walker-errors (walker subschema))
           c (or (coerce/json-coercion-matcher subschema) identity)]
       (fn [value]
         (-> value c walk))))
   schema))

(defn- deriver [schema]
  (start-walker
   (fn [subschema]
     (let [walk (walker subschema)
           dc (or (derive-coercion subschema) identity)]
       (fn [value]
         (-> value dc walk))))
   schema))

;; Deriving walker

(defn derive-schema [schema data]
  "Walk a schema, deriving any keys that are of the Derived type"
  (-> data
      ((json-coercer schema))
      ((deriver schema))))
