(ns libmooncat.ethereum.rpc
  (:require
   [clojure.string :as str]
   [libmooncat.data.contracts :as contracts]
   [libmooncat.data.resources :as resources]
   [libmooncat.ethereum.eth-util :as eth-util]
   [libmooncat.util]
   #?(:clj [libmooncat.ethereum.macros :refer [contract-function-data-builder with-result-data]]
      :cljs [libmooncat.ethereum.macros :refer [contract-function-data-builder]])
   #?(:cljs [libmooncat.ethereum.macros-cljs :refer [with-result-data]])
   [libmooncat.ethereum.http :refer [make-contract-request]]))

(defn- add-acclimator-details [rpc-url {:as result :keys [rescue-order]}]
  (with-result-data [data (make-contract-request rpc-url (:address contracts/mooncatacclimator)
                                                 ((contract-function-data-builder "ownerOf" [:uint256 token-id])
                                                  {:token-id rescue-order}))]
    (let [true-owner (eth-util/extract-address (eth-util/get-word data 0))]
      (assoc result
             :owner true-owner
             :is-acclimated true
             :contract {:token-id rescue-order
                        :description "MoonCatAcclimator"
                        :address (:address contracts/mooncatacclimator)
                        :capabilities ["ERC20"]}))))


(defn get-cat-contract-details [rpc-url cat-id]
  (with-result-data [data (make-contract-request rpc-url (:address contracts/mooncatrescue)
                                                 ((contract-function-data-builder "getCatDetails" [:bytes5 cat-id])
                                                  {:cat-id cat-id}))]
    (let [owner (eth-util/extract-address (eth-util/get-word data 1))
          result {:cat-id (eth-util/extract-bytes5 (eth-util/get-word data 0))
                  :rescue-order (resources/cat-id->rescue-order cat-id)
                  :is-acclimated false
                  :cat-name (eth-util/extract-bytes-to-string (eth-util/get-word data 2))}]
      (condp = owner
        (:address contracts/mooncatacclimator)
        (add-acclimator-details rpc-url result)

        (:address contracts/deprecated-unofficial-mooncat-wrapper)
        (assoc result
               :contract {:description "Unsupported Unofficial MoonCat Wrapper"
                          :address (:address contracts/deprecated-unofficial-mooncat-wrapper)
                          :warning "Unsupported Contract"})

        (assoc result
               :owner owner
               :contract {:token-id cat-id
                          :description "MoonCatRescue"
                          :address (:address contracts/mooncatrescue)
                          :capabilities ["ERC20" "ERC721" "ERC998"]})))))

(defn is-cat-acclimated [rpc-url cat-id]
  (with-result-data [data (make-contract-request rpc-url (:address contracts/mooncatrescue)
                                                 ((contract-function-data-builder "getCatDetails" [:bytes5 cat-id])
                                                  {:cat-id cat-id}))]
    (= (eth-util/extract-address (eth-util/get-word data 1))
       (:address contracts/mooncatacclimator))))
