package com.platform.http.executor;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.platform.http.OkHttpUtils;
import com.platform.http.ServiceEndpointProvider;
import lombok.Data;
import lombok.Singular;
import lombok.extern.slf4j.Slf4j;
import okhttp3.HttpUrl;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;

import java.nio.charset.Charset;
import java.util.List;

import static com.platform.http.executor.HttpUtils.checkNull;


/**
 * This is a generic flow execution
 * Basically a wrapper, that makes an http call on a client, using all the resources provided
 *
 * @author tushar.naik
 * @version 1.0  11/11/17 - 12:20 AM
 */
@Data
@Slf4j
public abstract class BaseHttpExecutor<T> implements HttpExecutor<T> {

    /* create and initialize this okhttp client */
    OkHttpClient client;

    /* this is used for logging and as a hystrix grouping key */
    String command;

    /* endpoint provider */
    ServiceEndpointProvider endpointProvider;

    /* final url to be hit */
    String url = null;

    /* list of header keyvalue pairs */
    @Singular("header")
    List<HeaderPair> headers = null;

    /* post data for http request */
    HttpData httpData;

    /* object mapper */
    ObjectMapper mapper;

    /* response type class */
    Class<T> responseType;

    /* response type reference class */
    TypeReference<T> responseTypeReference;

    /* if a secure http url needs to be used */
    boolean secure;

    Consumer<Exception, T> exceptionConsumer;

    Consumer<ExtractedResponse, T> nonSuccessResponseConsumer;

    BaseHttpExecutor(OkHttpClient client, String command,
                               ServiceEndpointProvider endpointProvider, String url,
                               List<HeaderPair> headers, HttpData httpData,
                               ObjectMapper mapper, Class<T> responseType, boolean secure,
                               Consumer<Exception, T> exceptionConsumer,
                               Consumer<ExtractedResponse, T> nonSuccessResponseConsumer,
                               TypeReference<T> responseTypeReference) {
        this.client = client;
        this.command = command;
        this.endpointProvider = endpointProvider;
        this.url = url;
        this.headers = headers;
        this.httpData = httpData;
        this.mapper = mapper;
        this.responseType = responseType;
        this.secure = secure;
        this.exceptionConsumer = exceptionConsumer;
        this.nonSuccessResponseConsumer = nonSuccessResponseConsumer;
        this.responseTypeReference = responseTypeReference;

    }

    RequestBody getRequestBody() throws JsonProcessingException {
        if (httpData.isSerialized()) {
            if (httpData.getData() instanceof String) {
                return RequestBody.create(MediaType.parse(httpData.getMediaType()), (String) httpData.getData());
            } else if (httpData.getData() instanceof byte[]) {
                return RequestBody.create(MediaType.parse(httpData.getMediaType()), (byte[]) httpData.getData());
            }
            throw new IllegalArgumentException("Illegal serialized flag");
        } else {
            return RequestBody.create(MediaType.parse(httpData.getMediaType()), mapper.writeValueAsBytes(httpData.getData()));
        }
    }

    public abstract Request getRequest(HttpUrl httpUrl) throws JsonProcessingException;

    public T execute() throws Exception {
        preconditions();
        Request request = null;
        try {
            final HttpUrl httpUrl = secure ?
                    HttpUtils.getSecureHttpUrl(endpointProvider, url) :
                    HttpUtils.getHttpUrl(endpointProvider, url);
            request = getRequest(httpUrl);
            log.info("Service {} request: {}", command, request);
            final Response response = client.newCall(request).execute();

            /* OkHttpUtils.body ensures that the response is closed */
            final byte[] responseBody = OkHttpUtils.body(response);
            if (!response.isSuccessful()) {
                if (nonSuccessResponseConsumer != null) {
                    return nonSuccessResponseConsumer.consume(extract(response, responseBody));
                }
                log.error(String.format("Service %s call failed statusCode:%d request:%s response:%s",
                                        command, response.code(), request, new String(responseBody)));
                throw new Exception("Service " + command + " call failure: " + response.code());
            }
            if (responseTypeReference != null) {
                return mapper.readValue(responseBody, responseTypeReference);
            }
            if (byte[].class.equals(responseType)) {
                return (T) responseBody;
            }
            if (String.class.equals(responseType)) {
                return (T) new String(responseBody, Charset.defaultCharset());
            }
            return mapper.readValue(responseBody, responseType);
        } catch (Exception e) {
            if (exceptionConsumer != null) {
                return exceptionConsumer.consume(e);
            }
            log.error(String.format("Error executing %s request:%s error:%s", command, request, e.getMessage()), e);
            throw e;
        }
    }

    @Override
    public String getCommand() {
        return command;
    }

    public void preconditions() {
        checkNull(client, "client cant be null");
        checkNull(endpointProvider, "endpointProvider cant be null");
        checkNull(url, "url cant be null");
        checkNull(mapper, "object mapper cant be null");
        checkNull(responseType, responseTypeReference, "both responseType and responseTypeReference cant be null");
    }

    private ExtractedResponse extract(Response response, byte[] responseBody) {
        return ExtractedResponse.builder().body(responseBody)
                                .code(response.code())
                                .headers(response.headers())
                                .message(response.message())
                                .protocol(response.protocol())
                                .handshake(response.handshake())
                                .receivedResponseAtMillis(response.receivedResponseAtMillis())
                                .sentRequestAtMillis(response.sentRequestAtMillis())
                                .build();
    }
}
