(ns membrane.parse
  (:require [instaparse.core :as insta]))

;; based on https://github.com/codemirror/CodeMirror/blob/858f5dcc2cf19d42393c360721e5c3a7dc547c43/mode/clojure/clojure.js

(def core-symbols (keys (ns-publics (the-ns 'clojure.core))))

(def special-forms (keys (. clojure.lang.Compiler specials)))
(def atoms ["false", "nil", "true"])

(def have-body-parameter [
                          "->", "->>", "as->", "binding", "bound-fn", "case", "catch", "comment",
                          "cond", "cond->", "cond->>", "condp", "def", "definterface", "defmethod",
                          "defn", "defmacro", "defprotocol", "defrecord", "defstruct", "deftype",
                          "do", "doseq", "dotimes", "doto", "extend", "extend-protocol",
                          "extend-type", "fn", "for", "future", "if", "if-let", "if-not", "if-some",
                          "let", "letfn", "locking", "loop", "ns", "proxy", "reify", "struct-map",
                          "some->", "some->>", "try", "when", "when-first", "when-let", "when-not",
                          "when-some", "while", "with-bindings", "with-bindings*", "with-in-str",
                          "with-loading-context", "with-local-vars", "with-meta", "with-open",
                          "with-out-str", "with-precision", "with-redefs", "with-redefs-fn"])

(def delimiter #"^(?:[\\\[\]\s\"(),;@^`{}~]|$)")
(def number-literal #"^(?:[+\-]?\d+(?:(?:N|(?:[eE][+\-]?\d+))|(?:\.?\d*(?:M|(?:[eE][+\-]?\d+))?)|\/\d+|[xX][0-9a-fA-F]+|r[0-9a-zA-Z]+)?(?=[\\\[\]\s\"#'(),;@^`{}~]|$))" )
(def character-literal #"^(?:\\(?:backspace|formfeed|newline|return|space|tab|o[0-7]{3}|u[0-9A-Fa-f]{4}|x[0-9A-Fa-f]{4}|.)?(?=[\\\[\]\s\"(),;@^`{}~]|$))")
(def qualified-symbol #"^(?:(?:[^\\\/\[\]\d\s\"#'(),;@^`{}~][^\\\[\]\s\"(),;@^`{}~]*(?:\.[^\\\/\[\]\d\s\"#'(),;@^`{}~][^\\\[\]\s\"(),;@^`{}~]*)*\/)?(?:\/|[^\\\/\[\]\d\s\"#'(),;@^`{}~][^\\\[\]\s\"(),;@^`{}~]*)*(?=[\\\[\]\s\"(),;@^`{}~]|$))")


(defn in-string [stream state]
  (loop [escaped? false
         c (.next stream)]
    (if (= "\"" c)
      [nil "string"]
      (recur false (.next stream)))))

(defn in-comment [stream state]
  )

(defn base [stream state]
  (cond
    ;; if (stream.eatSpace() || stream.eat(",")) return ["space", null];
    (or (.eatSpace stream)
        (.eat stream ","))
    ["space" nil]

    ;; if (stream.match(numberLiteral)) return [null, "number"];
    (.match stream number-literal)
    [nil "number"]
    
    ;; if (stream.match(characterLiteral)) return [null, "string-2"];
    (.match stream character-literal)
    [nil "string-2"]

    ;; if (stream.eat(/^"/)) return (state.tokenize = inString)(stream, state);
    (.eat stream #"^\"")
    (in-string stream (assoc state :tokenize in-string))

    ;; if (stream.eat(/^[(\[{]/)) return ["open", "bracket"];
    (.eat stream #"[(\[{]")
    ["open" "bracket"]
    
    ;; if (stream.eat(/^[)\]}]/)) return ["close", "bracket"];
    (.eat stream #"^[)\]}]")
    ["close" "bracket"]

    ;; if (stream.eat(/^;/)) {stream.skipToEnd(); return ["space", "comment"];}
    (.eat stream #"^;")
    (do
      (.skipToEnd stream)
      ["space" "comment"])
    
    ;; if (stream.eat(/^[#'@^`~]/)) return [null, "meta"];
    (.eat stream #"^[#'@^`~]")
    [nil "meta"]

    ;; var matches = stream.match(qualifiedSymbol);
    ;; var symbol = matches && matches[0];
    (.match stream qualified-symbol)
    nil
    

    ;; if (!symbol) {
    ;;   // advance stream by at least one character so we don't get stuck.
    ;;   stream.next();
    ;;   stream.eatWhile(function (c) {return !is(c, delimiter);});
    ;;   return [null, "error"];
    ;; }

    ;; if (symbol === "comment" && state.lastToken === "(")
    ;;   return (state.tokenize = inComment)(stream, state);
    ;; if (is(symbol, atom) || symbol.charAt(0) === ":") return ["symbol", "atom"];
    ;; if (is(symbol, specialForm) || is(symbol, coreSymbol)) return ["symbol", "keyword"];
    ;; if (state.lastToken === "(") return ["symbol", "builtin"]; // other operator

    ;; return ["symbol", "variable"];
    )

  )


(defn get-class [sym]
  (or (get {'char Character/TYPE
         'int Integer/TYPE}
        sym
        )
      (java.lang.Class/forName (name sym))))

(defmacro get-private-fn [cls private-name args]
  `(let [info# (->> (clojure.reflect/reflect ~cls)
                    :members
                    (filter #(= (:name %) (quote ~private-name)))
                    first)
        method# (doto (.getDeclaredMethod ~cls ~(name private-name) (into-array Class (map get-class (:parameter-types info#))))
                 (.setAccessible true))]

     (fn ~args
       (.invoke method# ~cls (to-array ~args)))))


(def read-token (get-private-fn clojure.lang.LispReader readToken [pbr initch]))
(def is-whitespace? (get-private-fn clojure.lang.LispReader isWhitespace [ch]))
(def read-number (get-private-fn clojure.lang.LispReader readNumber [pbr ch]))
(def unread (get-private-fn clojure.lang.LispReader unread [pbr ch]))
(def read1 (get-private-fn clojure.lang.LispReader read1 [pbr]))

;; (def read-token-method
;;   (doto (.getDeclaredMethod clojure.lang.LispReader "readToken" (into-array Class [java.io.PushbackReader Character/TYPE
;;                                                                                    ]))
;;     (.setAccessible true)))

;; (defn read-token [rdr c]
;;   (.invoke read-token-method clojure.lang.LispReader (to-array [rdr
;;                                                                 c])))

(def rdr (java.io.PushbackReader. (java.io.StringReader. "(foobar (+ a b \"ys\\\" asdf.\"))")))

(defn next-token [r]
  (let [ch (loop [ch (read1 r)]
             (if (is-whitespace? ch)
               (recur (read1 r))
               ch))]
    (when (not= -1 ch)
      (cond
        (Character/isDigit ch) (throw (Exception. "asdfsdf"))

        (or (= ch "+")
            (= ch "-")) (throw (Exception. "asdfsdf"))

        :else (read-token r (char ch))))))

(next-token rdr)



(def my-lang
  (insta/parser
    "
VEC = LPA EXPR+ RPA
LPA = \"(\"
RPA = \"(\"
     EXPR = #'[A-Za-z]+'
"))
(insta/parses my-lang "(Foo" :partial true)


(def text (slurp "/Users/adrian/workspace/membrane/docs/tutorial.md"))
(membrane.skia/copy-to-clipboard
 (->> text
      (clojure.string/split-lines)
      (keep #(re-matches #"^(#+)\s+(.*)" %))
      (map (fn [[_ hashes title]]
             (let [link (clojure.string/lower-case title)
                   link (clojure.string/replace link #" " "-")
                   link (clojure.string/replace link #"[^a-z\-]" "")]
               (str
               
                (apply str (repeat (* 4 (dec (count hashes))) " " ))
                "- "
                "[" title "]"
                "(" "/docs/tutorial.md#" link ")"
               
               
                ))))
      (clojure.string/join "\n")
      ))

