(ns membrane.example.counter
  (:require #?(:clj [membrane.skia :as skia])
            [membrane.ui :as ui
             :refer [vertical-layout
                     translate
                     horizontal-layout
                     button
                     label
                     with-color
                     bounds
                     spacer
                     on]]
            [membrane.component :as component
             :refer [defui run-ui run-ui-sync defeffect]]
            [membrane.basic-components :as basic])
  (:gen-class))


;; Display a "more!" button next to label showing num
;; clicking on "more!" will dispatch a ::counter-increment effect
(defui counter [& {:keys [num]}]
  (horizontal-layout
   (on :mouse-down (fn [[mouse-x mouse-y]]
                     [[::counter-increment $num]])
       (ui/button "more!"))
   (ui/label num)))

(defeffect ::counter-increment [$num]
  (dispatch! :update $num inc))

(comment
  ;; pop up a window showing our counter with
  ;; num initially set to 10
  (run-ui #'counter {:num 10}))

;; Display an "Add Counter" button
;; on top of a stack of counters
;;
;; clicking on the "Add Counter" button will
;; add a new counter to the bottom of the stack
;; 
;; clicking on the counters' "more!" buttons will
;; update their respective numbers
(defui counter-counter [& {:keys [nums]}]
  (apply
   vertical-layout
   (on :mouse-down (fn [[mx my]]
                     [[::add-counter $nums]])
       (ui/button "Add Counter"))
   (for [num nums]
     (counter :num num))))

(defeffect ::add-counter [$nums]
  (dispatch! :update $nums conj 0))

(comment
  ;; pop up a window showing our counter-counter
  ;; with nums initially set to [0 1 2]
  (run-ui #'counter-counter {:nums [0 1 2]}))


(ns membrane.example.counter
  (:require [membrane.skia :as skia]
            [membrane.ui :as ui
             :refer [vertical-layout
                     translate
                     horizontal-layout
                     button
                     label
                     with-color
                     bounds
                     spacer
                     on]]
            [membrane.component :as component
             :refer [defui run-ui run-ui-sync defeffect]]
            [membrane.basic-components :as basic])
  )

;; (skia/run #(ui/rectangle 10 10
;;                          ))
;; (skia/run #(ui/label "Hello" (ui/font "Menlo.ttc" 22)))

;; (skia/run #)

(ui/translate 10 10
              (on :mouse-down (fn [[mx my]]
                                ;; side effect
                                (println "hi" mx my)
                                
                                ;; return nil to prevent error messages
                                nil)
                  (ui/label "Hello")))





(def app-state (atom false))

(defn checkbox [checked?]
  (on
   :mouse-down
   (fn [_]
     (swap! app-state not)
     nil)
   (ui/label (if checked?
               "X"
               "O"))))

(defui checkbox [& {:keys [checked?]}]
  (on
   :mouse-down
   (fn [_]
     [[::toggle $checked?]])
   (ui/label (if checked?
               "X"
               "O"))))

(defeffect ::toggle [$checked?]
  (dispatch! :update $checked? not))

(defui checkbox-test [& {:keys [x y z]}]
  (vertical-layout
   (checkbox :checked? x)
   (checkbox :checked? y)
   (checkbox :checked? z)
   (ui/label
    (with-out-str (clojure.pprint/pprint
                   {:x x
                    :y y
                    :z z})))))

(comment (membrane.component/run-ui #'checkbox-test {:x false :y true :z false}))



(let [elem (ui/no-events
            (ui/button "Big Red Button"
                       (fn []
                         [[:self-destruct!]])))]
  (ui/mouse-down elem [20 20]))



(let [child-elem (ui/on :key-press
                        (fn [s]
                          [[:child-effect1 s]
                           [:child-effect2 s]])
                        (ui/label "child elem"))
      elem (ui/on
            :key-press (fn [s]
                        (if (= s ".")
                          [[:do-something-special]]
                          (ui/key-press child-elem s)
                          ))
            child-elem)]
  {"a" (ui/key-press elem "a")
   "." (ui/key-press elem ".")})


(let [lower-case-letters (set (map str "abcdefghijklmnopqrstuvwxyz"))
      child-elem (ui/on :key-press
                        (fn [s]
                          [[:child-effect1 s]
                           [:child-effect2 s]])
                        (ui/label "child elem"))
      elem (ui/on
            :key-press (fn [s]
                         (when (contains? lower-case-letters s)
                           (ui/key-press child-elem s)))
            child-elem)]
  {"a" (ui/key-press elem "a")
   "." (ui/key-press elem ".")})




(defn search-bar [s]
  (horizontal-layout
   (on
    :mouse-down (fn [_]
                  [[:search s]])
    (ui/button "Search"))
   (ui/label s)))
(let [selected-search-type :full-text
      bar (search-bar "clojure")
      elem (on :search
               (fn [s]
                 [[:search selected-search-type s]])
               bar)]
  (ui/mouse-down elem
                 [10 10]))
(def folder  (clojure.java.io/file "/Users/adrian/workspace/eddie/"))
(def fs (->> (.listFiles
              folder)
             (filter #(clojure.string/ends-with? (.getName %) ".mp3"))
             sort))
(defui opener [& {:keys [fs to-open]}]
  (let [to-open (or to-open #{})]
    (apply
     vertical-layout
     (basic/button :text "open"
             :on-click (fn []
                         (let [                                           
                               cmd (into-array String
                                               (into ["open"
                                                      "-a" "VLC"]
                                                     (map #(.getAbsolutePath %) to-open))
                                               )
                               
                               proc (.exec (Runtime/getRuntime) 
                                           cmd
                                           nil
                                           nil)]
                           (.close (.getOutputStream proc)))
                         nil))
     (for [f fs]
       (on
        :mouse-down (fn [_]
                        [[:update $to-open #(if (% f)
                                              (disj % f)
                                              (conj % f))]])
        (horizontal-layout
         (basic/checkbox :checked? (contains? to-open f))
         (ui/label (.getName f)))))))
  )
(comment
  (run-ui #'opener {:fs fs} ))




(defui item-row [ & {:keys [item-name selected?]}]
  (on
   :mouse-down
   (fn [_]
     [[:update $selected? not]])
   ;; put the items side by side
   (horizontal-layout
    (translate 5 5
               ;; checkbox in `membrane.ui` is non interactive.
               (ui/checkbox selected?))
    (spacer 5 0)
    (ui/label item-name))))

(comment
  (run-ui #'item-row {:item-name "my item" :selected? false}))



(defui item-selector
  "`item-names` a vector of choices to select from.
`selected` a set of selected items
`str-filter` filter out item names that don't contain a case insensitive match for `str-filter` as a substring
"
  [& {:keys [item-names selected str-filter]
      :or {str-filter ""
           selected #{}}}]
  (let [filter-fnames (filter #(clojure.string/includes? (clojure.string/lower-case %) str-filter) item-names)]
    (apply
     vertical-layout
     (basic/textarea :text str-filter)
     (for [fname filter-fnames]
       ;; override the default behaviour of updating the `selected?` value directly
       ;; instead, we'll keep the list of selected items in a set
       (on :update
           (fn [& args]
             [[:update $selected (fn [selected]
                                   (if (contains? selected fname)
                                     (disj selected fname)
                                     (conj selected fname)))]])
           (item-row :item-name fname :selected? (get selected fname)))))))

(comment
  (run-ui #'item-selector {:item-names (->> (clojure.java.io/file ".")
                                      (.listFiles)
                                      (map #(.getName %)))} ))


(defonce counter-state (atom 0))
(defn counter [num]
  (horizontal-layout
   (on :mouse-down (fn [[mouse-x mouse-y]]
                     (swap! counter-state inc)
                     nil)
       (button "more!"))
   (spacer 5 0)
   (label num (ui/font nil 19))))

(defui counter [& {:keys [num]}]
  (horizontal-layout
   (on :mouse-down (fn [[mouse-x mouse-y]]
                     [[::counter-increment $num]])
       (ui/button "more!"))
   (spacer 5 0)
   (ui/label num (ui/font nil 19))))
