(ns com.phronemophobic.clirun
  (:require
    ;; NOTE: ONLY depend on Clojure core, loaded in user's classpath so can't have any deps
    [clojure.edn :as edn]
    [clojure.java.io :as jio]
    [clojure.string :as str])
  (:import
    [clojure.lang ExceptionInfo]))


(set! *warn-on-reflection* true)

(defn- err
  ^Throwable [& msg]
  (ex-info (str/join " " msg) {:exec-msg true}))

(defn- requiring-resolve'
  ;; copied and modified from core to remove constraints on Clojure 1.10.x
  [sym]
  (if (nil? (namespace sym))
    (throw (err "Not a qualified symbol:" sym))
    (or (resolve sym)
      (do
        (-> sym namespace symbol require)
        (resolve sym)))))

#_(defn exec
  "Resolve and execute the function f (a symbol) with args"
  [f args]
  (let [resolved-f (requiring-resolve' f)]
    (if resolved-f
      (apply resolved-f args)
      (throw (err "Function not found:" f)))))

#_(defn- apply-overrides
  [args overrides]
  (reduce (fn [m [k v]]
            (if (sequential? k)
              (assoc-in m k v)
              (assoc m k v)))
    args (partition-all 2 overrides)))

#_(defn- qualify-fn
  "Compute function symbol based on exec-fn, ns-aliases, and ns-default"
  [fsym ns-aliases ns-default]
  ;; validation - make specs?
  (when (and fsym (not (symbol? fsym)))
    (throw (err "Expected function symbol:" fsym)))

  (when fsym
    (if (qualified-ident? fsym)
      (let [nsym (get ns-aliases (symbol (namespace fsym)))]
        (if nsym
          (symbol (str nsym) (name fsym))
          fsym))
      (if ns-default
        (symbol (str ns-default) (str fsym))
        (throw (err "Unqualified function can't be resolved:" fsym))))))

#_(defn- combine-alias-data
  "Combine the data from multiple aliases, for a particular key, given a combining rule"
  [alias-data key rule]
  (->> alias-data (map key) (remove nil?) rule))

#_(defn- resolve-alias
  "Retrieve an alias's data in basis"
  [basis alias]
  (get-in basis [:aliases alias]))

#_(defn- read-basis
  []
  (when-let [f (jio/file (System/getProperty "clojure.basis"))]
    (if (and f (.exists f))
      (->> f slurp (edn/read-string {:default tagged-literal}))
      (throw (err "No basis declared in clojure.basis system property")))))

#_(defn- read-aliases
  "Given some aliases, look up the aliases in the basis, combine the data per key,
  specifically the keys :exec-fn, :exec-args, :ns-aliases, and :ns-default.
  If :exec-args is an alias, resolve it."
  [basis aliases]
  (let [alias-data (map #(resolve-alias basis %) aliases)
        exec-args (combine-alias-data alias-data :exec-args #(apply merge %))
        resolved-args (if (keyword? exec-args) (resolve-alias basis exec-args) exec-args)]
    (when (not (or (nil? resolved-args) (map? resolved-args)))
      (throw (err "Invalid :exec-args, must be map or alias keyword:" resolved-args)))
    {:exec-fn (combine-alias-data alias-data :exec-fn last)
     :exec-args resolved-args
     :ns-aliases (combine-alias-data alias-data :ns-aliases #(apply merge %))
     :ns-default (combine-alias-data alias-data :ns-default last)}))

#_(defn- parse-fn
  [parsed [expr & exprs :as args]]
  (if (seq args)
    (if (odd? (count args))
      (if (symbol? expr)
        (cond-> (assoc parsed :function expr)
          (seq exprs) (assoc :overrides exprs))
        (throw (err "Key is missing value:" (last args))))
      (assoc parsed :overrides args))
    parsed))

#_(defn- parse-kws
  "Parses a concatenated string of keywords into a collection of keywords
  Ex: (parse-kws \":a:b:c\") ;; returns: (:a :b :c)"
  [s]
  (->> (str/split (or s "") #":")
    (remove str/blank?)
    (map
      #(if-let [i (str/index-of % \/)]
         (keyword (subs % 0 i) (subs % (inc i)))
         (keyword %)))))

#_(defn- parse-args
  [[a1 & as :as all]]
  (if (= a1 '--aliases)
    (parse-fn {:aliases (parse-kws (pr-str (first as)))} (rest as))
    (parse-fn nil all)))

(defn- read-args
  [args]
  (loop [[a & as] args
         read-args []]
    (if a
      (let [r (try
                (edn/read-string {:default tagged-literal} a)
                (catch Throwable _
                  (throw (err "Unreadable arg:" (pr-str a)))))]
        (recur as (conj read-args r)))
      read-args)))


#_(defn test-parse [& args]
  (let [{:keys [function aliases overrides]} (-> args read-args parse-args)
          {:keys [exec-fn exec-args ns-aliases ns-default]} (when aliases (read-aliases (read-basis) aliases))
          f (or function exec-fn)]
      [f ns-aliases ns-default exec-args overrides]
      #_(exec (qualify-fn f ns-aliases ns-default) (apply-overrides exec-args overrides))))

(defn no-args []
  (prn "hi"))

(defn test-fn [& args]
  (prn "args:" args))

(defn -main
  [& args]
  (try
    (let [[f & args] (read-args args)]
      (when (nil? f)
        (throw (err "No function found on command line")))
      (let [resolved-f (requiring-resolve' f)]

        (if resolved-f
          (apply resolved-f args)
          (throw (err "Function not found:" f)))))
    (catch ExceptionInfo e
      (if (-> e ex-data :exec-msg)
        (binding [*out* *err*]
          (println (.getMessage e))
          (System/exit 1))
        (throw e)))))




