(ns lein-lib.plugin
  "The main directory for the lein-lib plugin.
  For a project that includes this plugin, leiningen will load this namespace automagically and execute middleware over the project.

  The profile manipulations here (add-profile-paths, merged-profile, etc.,) are mostly copied from lein-monolith (https://github.com/amperity/lein-monolith/blob/master/src/lein_monolith/plugin.clj)"
  (:require [leiningen.core.main :as lein-main]
            [leiningen.core.project :as lein-project]))

(defn- distinct-by
  "Like distinct, but distinguishes elements by the value of (f x) for x in xs
  (distinct-by :x [{:x 1 :y 2} {:x 2 :y 2} {:x 1 :y 3}]) => [{:x 1 :y 2} {:x 2 :y 2}]"
  [f xs]
  (reverse (reduce (fn [acc x] (if (.contains (map f acc) (f x)) acc (cons x acc))) '() xs)))

(def ^:private ^:const monolith-directory
  "The root of the monolith from which we derive paths. E.g., /usr/local/phonewagon/mono. Does NOT include trailing slash."
  (clojure.string/trim (or "/usr/local/phonewagon/core-api"
                           (:out (clojure.java.shell/sh "git" "rev-parse" "--show-toplevel")))))

(defn- to-lib-directory
  "Takes a library name and returns the absolute path of its directory."
  [lib-name]
  (prn "HEYHEYHEY")
  (prn (:out (clojure.java.shell/sh "pwd")))
  (prn (:out (clojure.java.shell/sh "ls")))
  (str ;; monolith-directory
       "/usr/local/phonewagon/core-api" "/" lib-name))

(defn- to-lib-project-clj
  "Takes a library name and returns the absolute path to its project.clj"
  [lib-name]
  (str (to-lib-directory lib-name) "/" "project.clj"))

(defn- lib-str-to-project
  "Takes a library name and returns its leiningen-parsed project.clj"
  [lib-str]
  (lein-project/read-raw (to-lib-project-clj lib-str)))

(defn- add-profile-paths
  "Update a profile paths entry by adding the absolute paths from the given
  project. Returns the updated profile.

  Can optionally provide `with-root?`, which will determine whether we expand the paths
  based on the new project's root. `with-root?` should always be false if the underlying
  type of (get new-project k) is not a collection of strings."
  [base-project new-project k with-root?]
  (let [additions (if with-root?
                    (let [r (:root new-project)]
                      (map (partial str r "/") (get new-project k)))
                    (get new-project k))]
    (update base-project k (comp vec distinct into) additions)))


(defn- merged-profile
  "Constructs a project map containing merged (re)sources and dependencies."
  [project subprojects]
  (->
   (reduce
       (fn [project profile]
         (-> project
             (add-profile-paths profile :resource-paths true)
             (add-profile-paths profile :source-paths true)
             (add-profile-paths profile :dependencies false)))
       project
       subprojects)
   ;; TODO: uncomment if we ever use profiles. Then we do not need all the keys, just the modified ones
   #_(select-keys [:resource-paths :source-paths :dependencies])))

(def ^:private ^:const project-name
  "Given a leiningen-parsed project, returns that project's name" :name)

(defn- profile-active?
  "Check whether the given profile key is in the set of active profiles on the
  given project."
  [project profile-key]
  (contains? (set (:active-profiles (meta project))) profile-key))

;; TODO: uncomment if we want to use leiningen's profiles facility. Cribbed from lein-monolith (https://github.com/amperity/lein-monolith/blob/master/src/lein_monolith/plugin.clj)

;; (defn- add-profile
;;   "Adds the monolith profile to the given project if it's not already present."
;;   [project profile-key profile]
;;   (if (= profile (get-in project [:profiles profile-key]))
;;     project
;;     (do (lein-main/debug "Adding" profile-key "profile to project" (project-name project))
;;         (lein-project/add-profiles project {profile-key profile}))))

;; (defn- activate-profile
;;   "Activates the monolith profile in the project if it's not already active."
;;   [project profile-key]
;;   (if (profile-active? project profile-key)
;;     project
;;     (do (lein-main/debug "Merging" profile-key "profile into project" (project-name project))
;;         (lein-project/merge-profiles project [profile-key]))))

;; (defn- add-active-profile
;;   "Combines the effects of `add-profile` and `activate-profile`."
;;   [project profile-key profile]
;;   (-> project
;;       (add-profile profile-key profile)
;;       (activate-profile profile-key)))

(defn- gather-local-libs
  "Given a parsed project, return a full list of parsed local lib projects upon which that project depends.
  Traverses dependencies recursively and returns a flattened project list."
  [project]
  (loop [projects [project] accum-local-libs []]
    (if (empty? projects)
      accum-local-libs
      (let [local-libs-projects (map lib-str-to-project (:local-libs (first projects)))]
        (recur
         (concat (next projects) local-libs-projects)
         (distinct-by :name (concat accum-local-libs local-libs-projects)))))))

;; TODO: Figure out how to use profiles rather than directly merging / middleware
(defn middleware [project]
  (lein-main/debug "Applying lein-lib middleware to project " (project-name project))
  (if (profile-active? project :lein-lib/locals)
    (do (lein-main/debug "lein-lib profile already active in project " (project-name project))
        project)
    (if-let [libs (seq (gather-local-libs project))]
      (let [merged (merged-profile project libs)]
        (lein-main/debug (format "Found local-libs in project %s: %s" (project-name project) (map project-name libs)))
        (lein-project/add-profiles project {:lein-lib/locals merged})
        merged)
      (do (lein-main/debug "lein-lib found no local libs in project " (project-name project))
          project))))
